@extends('components.layouts.admin')

@section('title', 'Transfers')

@section('content')
    <div class="space-y-6">
        <!-- Success Message -->
        @if (session('success'))
            <x-admin.message type="success">
                {{ session('success') }}
            </x-admin.message>
        @endif

        <!-- Error Message -->
        @if (session('error'))
            <x-admin.message type="error">
                {{ session('error') }}
            </x-admin.message>
        @endif

        <!-- Header -->
        <div class="flex justify-between items-center">
            <div>
                <h1 class="text-2xl font-bold text-gray-900">Transfers</h1>
                <p class="text-gray-600">View all money transfers and their status</p>
            </div>
            <flux:button icon="plus" href="{{ route('admin.transfers.create') }}" variant="primary">
                Create Transfer
            </flux:button>
        </div>

        <!-- Filters -->
        <div class="bg-white rounded-xl border border-gray-200 shadow-sm p-6">
            <form method="GET" class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                <!-- Search -->
                <flux:field>
                    <flux:label>Search</flux:label>
                    <flux:input 
                        name="search" 
                        value="{{ request('search') }}" 
                        placeholder="Transaction ID, reference, description..."
                    />
                </flux:field>

                <!-- Status -->
                <flux:field>
                    <flux:label>Status</flux:label>
                    <flux:select name="status">
                        <option value="">All Statuses</option>
                        <option value="pending" {{ request('status') === 'pending' ? 'selected' : '' }}>Pending</option>
                        <option value="processing" {{ request('status') === 'processing' ? 'selected' : '' }}>Processing</option>
                        <option value="completed" {{ request('status') === 'completed' ? 'selected' : '' }}>Completed</option>
                        <option value="failed" {{ request('status') === 'failed' ? 'selected' : '' }}>Failed</option>
                        <option value="cancelled" {{ request('status') === 'cancelled' ? 'selected' : '' }}>Cancelled</option>
                        <option value="reversed" {{ request('status') === 'reversed' ? 'selected' : '' }}>Reversed</option>
                    </flux:select>
                </flux:field>

                <!-- Transfer Method -->
                <flux:field>
                    <flux:label>Transfer Method</flux:label>
                    <flux:select name="transfer_method">
                        <option value="">All Methods</option>
                        <option value="wire" {{ request('transfer_method') === 'wire' ? 'selected' : '' }}>Wire Transfer</option>
                        <option value="ach" {{ request('transfer_method') === 'ach' ? 'selected' : '' }}>ACH</option>
                        <option value="swift" {{ request('transfer_method') === 'swift' ? 'selected' : '' }}>SWIFT</option>
                        <option value="sepa" {{ request('transfer_method') === 'sepa' ? 'selected' : '' }}>SEPA</option>
                        <option value="instant" {{ request('transfer_method') === 'instant' ? 'selected' : '' }}>Instant</option>
                        <option value="domestic" {{ request('transfer_method') === 'domestic' ? 'selected' : '' }}>Domestic</option>
                    </flux:select>
                </flux:field>

                <!-- Transfer Speed -->
                <flux:field>
                    <flux:label>Transfer Speed</flux:label>
                    <flux:select name="transfer_speed">
                        <option value="">All Speeds</option>
                        <option value="instant" {{ request('transfer_speed') === 'instant' ? 'selected' : '' }}>Instant</option>
                        <option value="same_day" {{ request('transfer_speed') === 'same_day' ? 'selected' : '' }}>Same Day</option>
                        <option value="next_day" {{ request('transfer_speed') === 'next_day' ? 'selected' : '' }}>Next Day</option>
                        <option value="standard" {{ request('transfer_speed') === 'standard' ? 'selected' : '' }}>Standard</option>
                    </flux:select>
                </flux:field>

                <!-- Currency -->
                <flux:field>
                    <flux:label>Currency</flux:label>
                    <flux:select name="currency">
                        <option value="">All Currencies</option>
                        <option value="USD" {{ request('currency') === 'USD' ? 'selected' : '' }}>USD</option>
                        <option value="EUR" {{ request('currency') === 'EUR' ? 'selected' : '' }}>EUR</option>
                        <option value="GBP" {{ request('currency') === 'GBP' ? 'selected' : '' }}>GBP</option>
                        <option value="NGN" {{ request('currency') === 'NGN' ? 'selected' : '' }}>NGN</option>
                    </flux:select>
                </flux:field>

                <!-- User -->
                <flux:field>
                    <flux:label>User</flux:label>
                    <flux:select name="user_id">
                        <option value="">All Users</option>
                        @foreach($users as $user)
                            <option value="{{ $user->id }}" {{ request('user_id') == $user->id ? 'selected' : '' }}>
                                {{ $user->name }} ({{ $user->email }})
                            </option>
                        @endforeach
                    </flux:select>
                </flux:field>

                <!-- From Account -->
                <flux:field>
                    <flux:label>From Account</flux:label>
                    <flux:select name="from_account_id">
                        <option value="">All Accounts</option>
                        @foreach($accounts as $account)
                            <option value="{{ $account->id }}" {{ request('from_account_id') == $account->id ? 'selected' : '' }}>
                                {{ $account->account_number }} - {{ $account->account_name }}
                            </option>
                        @endforeach
                    </flux:select>
                </flux:field>

                <!-- To Account -->
                <flux:field>
                    <flux:label>To Account</flux:label>
                    <flux:select name="to_account_id">
                        <option value="">All Accounts</option>
                        @foreach($accounts as $account)
                            <option value="{{ $account->id }}" {{ request('to_account_id') == $account->id ? 'selected' : '' }}>
                                {{ $account->account_number }} - {{ $account->account_name }}
                            </option>
                        @endforeach
                    </flux:select>
                </flux:field>

                <!-- Beneficiary -->
                <flux:field>
                    <flux:label>Beneficiary</flux:label>
                    <flux:select name="beneficiary_id">
                        <option value="">All Beneficiaries</option>
                        @foreach($beneficiaries as $beneficiary)
                            <option value="{{ $beneficiary->id }}" {{ request('beneficiary_id') == $beneficiary->id ? 'selected' : '' }}>
                                {{ $beneficiary->name }} ({{ $beneficiary->email }})
                            </option>
                        @endforeach
                    </flux:select>
                </flux:field>

                <!-- Date From -->
                <flux:field>
                    <flux:label>Date From</flux:label>
                    <flux:date-picker 
                        name="date_from" 
                        value="{{ request('date_from') }}"
                    />
                </flux:field>

                <!-- Date To -->
                <flux:field>
                    <flux:label>Date To</flux:label>
                    <flux:date-picker 
                        name="date_to" 
                        value="{{ request('date_to') }}"
                    />
                </flux:field>

                <!-- Amount Min -->
                <flux:field>
                    <flux:label>Min Amount</flux:label>
                    <flux:input 
                        name="amount_min" 
                        type="number"
                        step="0.01"
                        value="{{ request('amount_min') }}" 
                        placeholder="0.00"
                    />
                </flux:field>

                <!-- Amount Max -->
                <flux:field>
                    <flux:label>Max Amount</flux:label>
                    <flux:input 
                        name="amount_max" 
                        type="number"
                        step="0.01"
                        value="{{ request('amount_max') }}" 
                        placeholder="0.00"
                    />
                </flux:field>

                <!-- Sort By -->
                <flux:field>
                    <flux:label>Sort By</flux:label>
                    <flux:select name="sort_by">
                        <option value="created_at" {{ request('sort_by') === 'created_at' ? 'selected' : '' }}>Created Date</option>
                        <option value="transaction_id" {{ request('sort_by') === 'transaction_id' ? 'selected' : '' }}>Transaction ID</option>
                        <option value="amount" {{ request('sort_by') === 'amount' ? 'selected' : '' }}>Amount</option>
                        <option value="status" {{ request('sort_by') === 'status' ? 'selected' : '' }}>Status</option>
                        <option value="transfer_method" {{ request('sort_by') === 'transfer_method' ? 'selected' : '' }}>Transfer Method</option>
                        <option value="transfer_speed" {{ request('sort_by') === 'transfer_speed' ? 'selected' : '' }}>Transfer Speed</option>
                        <option value="estimated_arrival" {{ request('sort_by') === 'estimated_arrival' ? 'selected' : '' }}>Estimated Arrival</option>
                        <option value="actual_arrival" {{ request('sort_by') === 'actual_arrival' ? 'selected' : '' }}>Actual Arrival</option>
                    </flux:select>
                </flux:field>

                <!-- Sort Direction -->
                <flux:field>
                    <flux:label>Sort Direction</flux:label>
                    <flux:select name="sort_direction">
                        <option value="desc" {{ request('sort_direction') === 'desc' ? 'selected' : '' }}>Descending</option>
                        <option value="asc" {{ request('sort_direction') === 'asc' ? 'selected' : '' }}>Ascending</option>
                    </flux:select>
                </flux:field>

                <!-- Filter Buttons -->
                <div class="flex gap-2">
                    <flux:button icon="magnifying-glass" type="submit" variant="primary">
                        Filter
                    </flux:button>
                    <flux:button icon="x-mark" href="{{ route('admin.transfers.index') }}" variant="outline">
                        Clear
                    </flux:button>
                </div>
            </form>
        </div>

        <!-- Transfers Table -->
        <div class="bg-white rounded-xl border border-gray-200 shadow-sm overflow-hidden">
            <div class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-200">
                    <thead class="bg-gray-50">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Transfer
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                User
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Method & Speed
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Amount
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Status
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Arrival
                            </th>
                            <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Actions
                            </th>
                        </tr>
                    </thead>
                    <tbody class="bg-white divide-y divide-gray-200">
                        @forelse($transfers as $transfer)
                            <tr class="hover:bg-gray-50">
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="flex items-center">
                                        <div>
                                            <div class="text-sm font-medium text-gray-900">
                                                {{ $transfer->transaction_id }}
                                            </div>
                                            <div class="text-sm text-gray-500">
                                                {{ $transfer->reference_number }}
                                            </div>
                                            @if($transfer->external_reference)
                                                <div class="text-xs text-gray-400">
                                                    Ext: {{ $transfer->external_reference }}
                                                </div>
                                            @endif
                                        </div>
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="text-sm text-gray-900">
                                        {{ $transfer->user->name }}
                                    </div>
                                    <div class="text-sm text-gray-500">
                                        {{ $transfer->user->email }}
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-blue-100 text-blue-800">
                                        {{ ucfirst($transfer->transfer_method ?? 'N/A') }}
                                    </span>
                                    <div class="text-sm text-gray-500 mt-1">
                                        {{ ucfirst($transfer->transfer_speed ?? 'N/A') }}
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="text-sm font-medium text-gray-900">
                                        {{ $transfer->currency }}{{ number_format($transfer->amount, 2) }}
                                    </div>
                                    @if($transfer->fee_amount > 0)
                                        <div class="text-sm text-gray-500">
                                            Fee: {{ $transfer->currency }}{{ number_format($transfer->fee_amount, 2) }}
                                        </div>
                                    @endif
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium
                                        @if($transfer->status === 'completed') bg-green-100 text-green-800
                                        @elseif($transfer->status === 'pending') bg-yellow-100 text-yellow-800
                                        @elseif($transfer->status === 'processing') bg-blue-100 text-blue-800
                                        @elseif($transfer->status === 'failed') bg-red-100 text-red-800
                                        @elseif($transfer->status === 'cancelled') bg-gray-100 text-gray-800
                                        @elseif($transfer->status === 'reversed') bg-purple-100 text-purple-800
                                        @else bg-gray-100 text-gray-800
                                        @endif">
                                        {{ ucfirst($transfer->status) }}
                                    </span>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                    @if($transfer->actual_arrival)
                                        <div class="text-green-600 font-medium">
                                            {{ $transfer->actual_arrival->format('M d, Y') }}
                                        </div>
                                        <div class="text-xs text-gray-400">
                                            {{ $transfer->actual_arrival->format('g:i A') }}
                                        </div>
                                    @elseif($transfer->estimated_arrival)
                                        <div class="text-blue-600">
                                            {{ $transfer->estimated_arrival->format('M d, Y') }}
                                        </div>
                                        <div class="text-xs text-gray-400">
                                            {{ $transfer->estimated_arrival->format('g:i A') }}
                                        </div>
                                    @else
                                        <span class="text-gray-400">Not set</span>
                                    @endif
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                                    <div class="flex items-center justify-end space-x-2">
                                        <flux:button 
                                            href="{{ route('admin.transfers.show', $transfer) }}" 
                                            variant="outline" 
                                            size="sm"
                                        >
                                            <flux:icon name="eye" class="w-4 h-4" />
                                        </flux:button>
                                        <flux:button 
                                            href="{{ route('admin.transfers.edit', $transfer) }}" 
                                            variant="outline" 
                                            size="sm"
                                        >
                                            <flux:icon name="pencil" class="w-4 h-4" />
                                        </flux:button>
                                        
                                        @if($transfer->status === 'pending')
                                            <form method="POST" action="{{ route('admin.transfers.approve', $transfer) }}" class="inline">
                                                @csrf
                                                <flux:button type="submit" variant="outline" size="sm" class="text-green-600 hover:text-green-700">
                                                    <flux:icon name="check" class="w-4 h-4" />
                                                </flux:button>
                                            </form>
                                            <form method="POST" action="{{ route('admin.transfers.reject', $transfer) }}" class="inline">
                                                @csrf
                                                <flux:button type="submit" variant="outline" size="sm" class="text-red-600 hover:text-red-700">
                                                    <flux:icon name="x-mark" class="w-4 h-4" />
                                                </flux:button>
                                            </form>
                                        @endif

                                        @if($transfer->status === 'processing')
                                            <form method="POST" action="{{ route('admin.transfers.mark-arrived', $transfer) }}" class="inline">
                                                @csrf
                                                <flux:button type="submit" variant="outline" size="sm" class="text-blue-600 hover:text-blue-700">
                                                    <flux:icon name="check-circle" class="w-4 h-4" />
                                                </flux:button>
                                            </form>
                                        @endif

                                        @if($transfer->status === 'completed')
                                            <form method="POST" action="{{ route('admin.transfers.reverse', $transfer) }}" class="inline">
                                                @csrf
                                                <flux:button type="submit" variant="outline" size="sm" class="text-purple-600 hover:text-purple-700">
                                                    <flux:icon name="arrow-uturn-left" class="w-4 h-4" />
                                                </flux:button>
                                            </form>
                                        @endif

                                        <form method="POST" action="{{ route('admin.transfers.destroy', $transfer) }}" class="inline" onsubmit="return confirm('Are you sure you want to delete this transfer? This action cannot be undone.')">
                                            @csrf
                                            @method('DELETE')
                                            <flux:button type="submit" variant="outline" size="sm" class="text-red-600 hover:text-red-700">
                                                <flux:icon name="trash" class="w-4 h-4" />
                                            </flux:button>
                                        </form>
                                    </div>
                                </td>
                            </tr>
                        @empty
                            <tr>
                                <td colspan="7" class="px-6 py-12 text-center">
                                    <div class="text-gray-500">
                                        <flux:icon name="inbox" class="w-12 h-12 mx-auto mb-4 text-gray-300" />
                                        <p class="text-lg font-medium">No transfers found</p>
                                        <p class="text-sm">Try adjusting your search criteria or create a new transfer.</p>
                                    </div>
                                </td>
                            </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>

            <!-- Pagination -->
            @if($transfers->hasPages())
                <div class="px-6 py-4 border-t border-gray-200">
                    {{ $transfers->links() }}
                </div>
            @endif
        </div>
    </div>
@endsection
