@extends('components.layouts.admin')

@section('title', 'Notification Settings')

@section('content')
<div class="space-y-6">
    <!-- Success Message -->
    @if (session('success'))
        <x-admin.message type="success">
            {{ session('success') }}
        </x-admin.message>
    @endif

    <!-- Error Message -->
    @if (session('error'))
        <x-admin.message type="error">
            {{ session('error') }}
        </x-admin.message>
    @endif

    <!-- Header -->
    <div class="flex justify-between items-center">
        <div>
            <h1 class="text-2xl font-bold text-gray-900">Notification Settings</h1>
            <p class="text-gray-600">Configure email verification and SMTP settings</p>
        </div>
        <div class="flex space-x-3">
            <flux:button href="{{ route('admin.settings.index') }}" variant="outline" class="text-xs">
                <flux:icon name="arrow-left" class="w-4 h-4 mr-2" />
                Back to Settings
            </flux:button>
        </div>
    </div>

    <form method="POST" action="{{ route('admin.settings.notifications.update') }}" class="space-y-6">
        @csrf
        
        <!-- Email Verification Status -->
        <flux:card>
            <flux:heading size="lg" class="mb-4">Email Verification</flux:heading>
            <div class="space-y-4">
                <flux:field>
                    <flux:label for="email_verification_enabled">Email Verification Status</flux:label>
                    <flux:select id="email_verification_enabled" name="email_verification_enabled" required>
                        <option value="1" {{ old('email_verification_enabled', $settings['email_verification_enabled'] ?? true) ? 'selected' : '' }}>Enabled</option>
                        <option value="0" {{ !old('email_verification_enabled', $settings['email_verification_enabled'] ?? true) ? 'selected' : '' }}>Disabled</option>
                    </flux:select>
                    <flux:error name="email_verification_enabled" />
                    <flux:description>Enable or disable email verification for new user registrations</flux:description>
                </flux:field>
            </div>
        </flux:card>

        <!-- SMTP Configuration -->
        <flux:card>
            <flux:heading size="lg" class="mb-4">SMTP Configuration</flux:heading>
            <div class="space-y-6">
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <flux:field>
                        <flux:label for="smtp_driver">SMTP Driver</flux:label>
                        <flux:select id="smtp_driver" name="smtp_driver" required>
                            <option value="smtp" {{ old('smtp_driver', $settings['smtp_driver'] ?? 'smtp') === 'smtp' ? 'selected' : '' }}>SMTP</option>
                            <option value="mailgun" {{ old('smtp_driver', $settings['smtp_driver']) === 'mailgun' ? 'selected' : '' }}>Mailgun</option>
                            <option value="ses" {{ old('smtp_driver', $settings['smtp_driver']) === 'ses' ? 'selected' : '' }}>Amazon SES</option>
                            <option value="postmark" {{ old('smtp_driver', $settings['smtp_driver']) === 'postmark' ? 'selected' : '' }}>Postmark</option>
                        </flux:select>
                        <flux:error name="smtp_driver" />
                        <flux:description>Choose your email service provider</flux:description>
                    </flux:field>

                    <flux:field>
                        <flux:label for="smtp_host">SMTP Host</flux:label>
                        <flux:input 
                            id="smtp_host" 
                            name="smtp_host" 
                            value="{{ old('smtp_host', $settings['smtp_host'] ?? 'smtp.gmail.com') }}"
                            placeholder="smtp.gmail.com"
                            required
                        />
                        <flux:error name="smtp_host" />
                        <flux:description>SMTP server hostname</flux:description>
                    </flux:field>
                </div>

                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <flux:field>
                        <flux:label for="smtp_port">SMTP Port</flux:label>
                        <flux:input 
                            id="smtp_port" 
                            name="smtp_port" 
                            type="number"
                            min="1"
                            max="65535"
                            value="{{ old('smtp_port', $settings['smtp_port'] ?? 587) }}"
                            placeholder="587"
                            required
                        />
                        <flux:error name="smtp_port" />
                        <flux:description>SMTP server port (587 for TLS, 465 for SSL)</flux:description>
                    </flux:field>

                    <flux:field>
                        <flux:label for="smtp_username">SMTP Username</flux:label>
                        <flux:input 
                            id="smtp_username" 
                            name="smtp_username" 
                            value="{{ old('smtp_username', $settings['smtp_username']) }}"
                            placeholder="your-email@gmail.com"
                            required
                        />
                        <flux:error name="smtp_username" />
                        <flux:description>Email address for SMTP authentication</flux:description>
                    </flux:field>
                </div>

                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <flux:field>
                        <flux:label for="smtp_password">SMTP Password</flux:label>
                        <flux:input 
                            id="smtp_password" 
                            name="smtp_password" 
                            type="password"
                            value="{{ old('smtp_password', $settings['smtp_password']) }}"
                            placeholder="Your email password or app password"
                            required
                        />
                        <flux:error name="smtp_password" />
                        <flux:description>Password or app-specific password for SMTP</flux:description>
                    </flux:field>

                    <flux:field>
                        <flux:label for="smtp_encryption">Encryption</flux:label>
                        <flux:select id="smtp_encryption" name="smtp_encryption">
                            <option value="tls" {{ old('smtp_encryption', $settings['smtp_encryption'] ?? 'tls') === 'tls' ? 'selected' : '' }}>TLS</option>
                            <option value="ssl" {{ old('smtp_encryption', $settings['smtp_encryption']) === 'ssl' ? 'selected' : '' }}>SSL</option>
                            <option value="" {{ old('smtp_encryption', $settings['smtp_encryption']) === '' ? 'selected' : '' }}>None</option>
                        </flux:select>
                        <flux:error name="smtp_encryption" />
                        <flux:description>Encryption method for SMTP connection</flux:description>
                    </flux:field>
                </div>

                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <flux:field>
                        <flux:label for="from_email">From Email</flux:label>
                        <flux:input 
                            id="from_email" 
                            name="from_email" 
                            type="email"
                            value="{{ old('from_email', $settings['from_email'] ?? config('mail.from.address')) }}"
                            placeholder="noreply@bank.cycloverseworld.info"
                            required
                        />
                        <flux:error name="from_email" />
                        <flux:description>Default sender email address</flux:description>
                    </flux:field>

                    <flux:field>
                        <flux:label for="from_name">From Name</flux:label>
                        <flux:input 
                            id="from_name" 
                            name="from_name" 
                            value="{{ old('from_name', $settings['from_name'] ?? config('mail.from.name')) }}"
                            placeholder="Cycloverse Bank"
                            required
                        />
                        <flux:error name="from_name" />
                        <flux:description>Default sender name</flux:description>
                    </flux:field>
                </div>

                <!-- Test SMTP Configuration -->
                <div class="flex justify-end">
                    <flux:button type="button" onclick="testSmtpConfiguration()" variant="outline" class="text-xs">
                        <flux:icon name="envelope" class="w-4 h-4 mr-2" />
                        Test SMTP Configuration
                    </flux:button>
                </div>
            </div>
        </flux:card>

        <!-- Current Status -->
        <flux:card>
            <flux:heading size="lg" class="mb-4">Current Status</flux:heading>
            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div class="text-center p-4 {{ ($settings['email_verification_enabled'] ?? true) ? 'bg-green-50' : 'bg-red-50' }} rounded-lg">
                    <flux:icon name="envelope" class="w-8 h-8 mx-auto {{ ($settings['email_verification_enabled'] ?? true) ? 'text-green-600' : 'text-red-600' }}" />
                    <div class="mt-2 text-sm font-medium {{ ($settings['email_verification_enabled'] ?? true) ? 'text-green-800' : 'text-red-800' }}">
                        Email Verification {{ ($settings['email_verification_enabled'] ?? true) ? 'Enabled' : 'Disabled' }}
                    </div>
                </div>
                <div class="text-center p-4 bg-blue-50 rounded-lg">
                    <flux:icon name="cog-6-tooth" class="w-8 h-8 mx-auto text-blue-600" />
                    <div class="mt-2 text-sm font-medium text-blue-800">
                        Driver: {{ ucfirst($settings['smtp_driver'] ?? 'smtp') }}
                    </div>
                    <div class="mt-1 text-xs text-blue-600">
                        {{ $settings['smtp_host'] ?? 'Not configured' }}
                    </div>
                </div>
            </div>
        </flux:card>

        <!-- Actions -->
        <div class="flex justify-between items-center">
            <flux:button type="button" variant="outline" onclick="resetForm()" class="text-xs">
                <flux:icon name="arrow-path" class="w-4 h-4 mr-2" />
                Reset Form
            </flux:button>
            
            <div class="flex space-x-3">
                <flux:button type="button" variant="outline" href="{{ route('admin.settings.index') }}" class="text-xs">
                    Cancel
                </flux:button>
                <flux:button icon="check" type="submit" class="text-xs">
                    Save Notification Settings
                </flux:button>
            </div>
        </div>
    </form>
</div>

<script>
function resetForm() {
    if (confirm('Are you sure you want to reset the form? All unsaved changes will be lost.')) {
        document.querySelector('form').reset();
    }
}

function testSmtpConfiguration() {
    const smtpDriver = document.getElementById('smtp_driver').value;
    const smtpHost = document.getElementById('smtp_host').value;
    const smtpPort = document.getElementById('smtp_port').value;
    const smtpUsername = document.getElementById('smtp_username').value;
    const smtpPassword = document.getElementById('smtp_password').value;
    const smtpEncryption = document.getElementById('smtp_encryption').value;
    const fromEmail = document.getElementById('from_email').value;
    const fromName = document.getElementById('from_name').value;

    if (!smtpHost || !smtpPort || !smtpUsername || !smtpPassword || !fromEmail || !fromName) {
        alert('Please fill in all required SMTP configuration fields before testing.');
        return;
    }

    // Show loading state
    const button = event.target;
    const originalText = button.innerHTML;
    button.innerHTML = '<svg class="w-4 h-4 mr-2 animate-spin" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"></path></svg>Testing...';
    button.disabled = true;

    // Get CSRF token
    const csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') 
                      || document.querySelector('input[name="_token"]')?.value;

    fetch('/admin/settings/test-smtp', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': csrfToken
        },
        body: JSON.stringify({
            smtp_driver: smtpDriver,
            smtp_host: smtpHost,
            smtp_port: parseInt(smtpPort),
            smtp_username: smtpUsername,
            smtp_password: smtpPassword,
            smtp_encryption: smtpEncryption,
            from_email: fromEmail,
            from_name: fromName
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert('✅ SMTP configuration test successful!\n\n' + data.message);
        } else {
            alert('❌ SMTP configuration test failed!\n\n' + data.message);
        }
    })
    .catch(error => {
        alert('❌ SMTP configuration test failed!\n\nError: ' + error.message);
    })
    .finally(() => {
        // Restore button state
        button.innerHTML = originalText;
        button.disabled = false;
    });
}
</script>
@endsection