@extends('components.layouts.admin')

@section('title', 'Permissions Management')

@section('content')
<div class="space-y-6">
    <!-- Success Message -->
    @if (session('success'))
        <x-admin.message type="success">
            {{ session('success') }}
        </x-admin.message>
    @endif

    <!-- Error Message -->
    @if (session('error'))
        <x-admin.message type="error">
            {{ session('error') }}
        </x-admin.message>
    @endif

    <!-- Header -->
    <div class="flex justify-between items-center">
        <div>
            <h1 class="text-2xl font-bold text-gray-900">Permissions Management</h1>
            <p class="text-gray-600">Manage system permissions and their assignments</p>
        </div>
        <div class="flex space-x-3">
            <flux:button icon="shield-check" href="{{ route('admin.roles.index') }}" variant="outline" class="text-xs">
                Roles
            </flux:button>
            <flux:button icon="plus" href="{{ route('admin.permissions.create') }}" class="text-xs">
                Create Permission
            </flux:button>
        </div>
    </div>

    <!-- Filters -->
    <flux:card>
        <form method="GET" class="space-y-4">
            <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                <flux:field>
                    <flux:label>Search</flux:label>
                    <flux:input 
                        name="search" 
                        value="{{ request('search') }}"
                        placeholder="Search by permission name..."
                    />
                </flux:field>

                <flux:field>
                    <flux:label>Category</flux:label>
                    <flux:select name="category">
                        <option value="">All Categories</option>
                        @foreach($categories as $category)
                            <option value="{{ $category }}" {{ request('category') === $category ? 'selected' : '' }}>
                                {{ ucfirst(str_replace('-', ' ', $category)) }}
                            </option>
                        @endforeach
                    </flux:select>
                </flux:field>

                <flux:field>
                    <flux:label>Sort By</flux:label>
                    <flux:select name="sort_by">
                        <option value="name" {{ request('sort_by') === 'name' ? 'selected' : '' }}>Name</option>
                        <option value="created_at" {{ request('sort_by') === 'created_at' ? 'selected' : '' }}>Created Date</option>
                        <option value="roles_count" {{ request('sort_by') === 'roles_count' ? 'selected' : '' }}>Roles Count</option>
                        <option value="users_count" {{ request('sort_by') === 'users_count' ? 'selected' : '' }}>Users Count</option>
                    </flux:select>
                </flux:field>
            </div>

            <div class="flex justify-between items-center">
                <flux:button icon="magnifying-glass" type="submit" variant="outline" class="text-xs">
                    Apply Filters
                </flux:button>
                <flux:button icon="x-mark" href="{{ route('admin.permissions.index') }}" variant="outline" class="text-xs">
                    Clear Filters
                </flux:button>
            </div>
        </form>
    </flux:card>

    <!-- Permissions Table -->
    <flux:card>
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            <a href="{{ request()->fullUrlWithQuery(['sort_by' => 'name', 'sort_direction' => request('sort_direction') === 'asc' ? 'desc' : 'asc']) }}" class="flex items-center space-x-1">
                                Permission Name
                                <flux:icon name="chevron-up-down" class="w-4 h-4" />
                            </a>
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Category</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            <a href="{{ request()->fullUrlWithQuery(['sort_by' => 'roles_count', 'sort_direction' => request('sort_direction') === 'asc' ? 'desc' : 'asc']) }}" class="flex items-center space-x-1">
                                Roles
                                <flux:icon name="chevron-up-down" class="w-4 h-4" />
                            </a>
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            <a href="{{ request()->fullUrlWithQuery(['sort_by' => 'users_count', 'sort_direction' => request('sort_direction') === 'asc' ? 'desc' : 'asc']) }}" class="flex items-center space-x-1">
                                Users
                                <flux:icon name="chevron-up-down" class="w-4 h-4" />
                            </a>
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Guard</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            <a href="{{ request()->fullUrlWithQuery(['sort_by' => 'created_at', 'sort_direction' => request('sort_direction') === 'asc' ? 'desc' : 'asc']) }}" class="flex items-center space-x-1">
                                Created
                                <flux:icon name="chevron-up-down" class="w-4 h-4" />
                            </a>
                        </th>
                        <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    @forelse($permissions as $permission)
                        @php
                            $category = explode('-', $permission->name)[0];
                        @endphp
                        <tr class="hover:bg-gray-50">
                            <td class="px-6 py-4 whitespace-nowrap">
                                <div class="flex items-center">
                                    <div class="flex-shrink-0 h-10 w-10">
                                        <div class="h-10 w-10 rounded-lg bg-gradient-to-r from-green-500 to-blue-600 flex items-center justify-center">
                                            <flux:icon name="key" class="w-6 h-6 text-white" />
                                        </div>
                                    </div>
                                    <div class="ml-4">
                                        <div class="text-sm font-medium text-gray-900">
                                            {{ ucfirst(str_replace('-', ' ', $permission->name)) }}
                                        </div>
                                        <div class="text-sm text-gray-500">{{ $permission->name }}</div>
                                    </div>
                                </div>
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap">
                                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-gray-100 text-gray-800 capitalize">
                                    {{ str_replace('-', ' ', $category) }}
                                </span>
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap">
                                <div class="flex items-center">
                                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800">
                                        {{ $permission->roles_count }} roles
                                    </span>
                                </div>
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap">
                                <div class="flex items-center">
                                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                                        {{ $permission->users_count }} users
                                    </span>
                                </div>
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                {{ $permission->guard_name }}
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                {{ $permission->created_at->format('M d, Y') }}
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                                <div class="flex items-center justify-end space-x-2">
                                    <flux:button href="{{ route('admin.permissions.show', $permission) }}" variant="outline" size="sm">
                                        <flux:icon name="eye" class="w-4 h-4" />
                                    </flux:button>
                                    <flux:button href="{{ route('admin.permissions.edit', $permission) }}" variant="outline" size="sm">
                                        <flux:icon name="pencil" class="w-4 h-4" />
                                    </flux:button>
                                    
                                    @if($permission->roles_count == 0)
                                        <form method="POST" action="{{ route('admin.permissions.destroy', $permission) }}" class="inline" onsubmit="return confirm('Are you sure you want to delete this permission?')">
                                            @csrf
                                            @method('DELETE')
                                            <flux:button type="submit" variant="outline" size="sm" class="text-red-600 hover:text-red-700">
                                                <flux:icon name="trash" class="w-4 h-4" />
                                            </flux:button>
                                        </form>
                                    @else
                                        <flux:button disabled variant="outline" size="sm" class="text-gray-400 cursor-not-allowed" title="Cannot delete permission assigned to roles">
                                            <flux:icon name="trash" class="w-4 h-4" />
                                        </flux:button>
                                    @endif
                                </div>
                            </td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="7" class="px-6 py-12 text-center text-gray-500">
                                <flux:icon name="key" class="w-12 h-12 mx-auto mb-4 text-gray-300" />
                                <p class="text-lg font-medium">No permissions found</p>
                                <p class="text-sm">Get started by creating a new permission.</p>
                            </td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>

        @if($permissions->hasPages())
            <div class="px-6 py-4 border-t border-gray-200">
                {{ $permissions->links() }}
            </div>
        @endif
    </flux:card>
</div>
@endsection
