@extends('components.layouts.admin')

@section('title', 'Log File - ' . $filename)

@section('content')
<div class="space-y-6">
    <!-- Success Message -->
    @if (session('success'))
        <x-admin.message type="success">
            {{ session('success') }}
        </x-admin.message>
    @endif

    <!-- Error Message -->
    @if (session('error'))
        <x-admin.message type="error">
            {{ session('error') }}
        </x-admin.message>
    @endif

    <!-- Header -->
    <div class="flex justify-between items-center">
        <div>
            <h1 class="text-2xl font-bold text-gray-900">Log File</h1>
            <p class="text-gray-600">{{ $filename }}</p>
        </div>
        <div class="flex space-x-3">
            <flux:button href="{{ route('admin.logs.index') }}" variant="outline" class="text-xs">
                <flux:icon name="arrow-left" class="w-4 h-4 mr-2" />
                Back to Logs
            </flux:button>
            <flux:button href="{{ route('admin.logs.download', $filename) }}" variant="outline" class="text-xs">
                <flux:icon name="arrow-down-tray" class="w-4 h-4 mr-2" />
                Download
            </flux:button>
            <form method="POST" action="{{ route('admin.logs.clear', $filename) }}" class="inline" onsubmit="return confirm('Are you sure you want to clear this log file?')">
                @csrf
                @method('DELETE')
                <flux:button icon="trash" type="submit" variant="outline" class="text-xs text-red-600 hover:text-red-700">
                    Clear Log
                </flux:button>
            </form>
        </div>
    </div>

    <!-- Log File Information -->
    <flux:card>
        <flux:heading size="lg" class="mb-4">File Information</flux:heading>
        <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">File Name</label>
                <p class="text-sm font-mono text-gray-900">{{ $filename }}</p>
            </div>
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">File Size</label>
                <p class="text-sm text-gray-900">{{ $logFiles->firstWhere('name', $filename)['size'] ?? 'Unknown' }}</p>
            </div>
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Last Modified</label>
                <p class="text-sm text-gray-900">{{ $logFiles->firstWhere('name', $filename)['modified'] ? date('M d, Y H:i', $logFiles->firstWhere('name', $filename)['modified']) : 'Unknown' }}</p>
            </div>
        </div>
    </flux:card>

    <!-- Filters -->
    <flux:card>
        <flux:heading size="lg" class="mb-4">Filters</flux:heading>
        <form method="GET" class="space-y-4">
            <input type="hidden" name="file" value="{{ $filename }}">
            
            <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                <flux:field>
                    <flux:label>Search</flux:label>
                    <flux:input 
                        name="search" 
                        value="{{ request('search') }}"
                        placeholder="Search log entries..."
                    />
                </flux:field>

                <flux:field>
                    <flux:label>Level</flux:label>
                    <flux:select name="level">
                        <option value="">All Levels</option>
                        <option value="emergency" {{ request('level') === 'emergency' ? 'selected' : '' }}>Emergency</option>
                        <option value="alert" {{ request('level') === 'alert' ? 'selected' : '' }}>Alert</option>
                        <option value="critical" {{ request('level') === 'critical' ? 'selected' : '' }}>Critical</option>
                        <option value="error" {{ request('level') === 'error' ? 'selected' : '' }}>Error</option>
                        <option value="warning" {{ request('level') === 'warning' ? 'selected' : '' }}>Warning</option>
                        <option value="notice" {{ request('level') === 'notice' ? 'selected' : '' }}>Notice</option>
                        <option value="info" {{ request('level') === 'info' ? 'selected' : '' }}>Info</option>
                        <option value="debug" {{ request('level') === 'debug' ? 'selected' : '' }}>Debug</option>
                    </flux:select>
                </flux:field>

                <flux:field>
                    <flux:label>Actions</flux:label>
                    <div class="flex space-x-2">
                        <flux:button icon="magnifying-glass" type="submit" variant="outline" size="sm" class="flex-1">
                            Filter
                        </flux:button>
                        <flux:button icon="x-mark" href="{{ route('admin.logs.show', $filename) }}" variant="outline" size="sm" class="flex-1">
                            Clear
                        </flux:button>
                    </div>
                </flux:field>
            </div>
        </form>
    </flux:card>

    <!-- Log Entries -->
    <flux:card>
        <flux:heading size="lg" class="mb-4">Log Entries</flux:heading>
        
        @if($logs->count() > 0)
            <!-- Log Statistics -->
            <div class="mb-4 p-4 bg-gray-50 rounded-lg">
                <div class="grid grid-cols-2 md:grid-cols-4 gap-4 text-center">
                    <div>
                        <div class="text-2xl font-bold text-gray-900">{{ $logs->count() }}</div>
                        <div class="text-sm text-gray-500">Current Page</div>
                    </div>
                    <div>
                        <div class="text-2xl font-bold text-gray-900">{{ $pagination['total'] ?? 0 }}</div>
                        <div class="text-sm text-gray-500">Total Entries</div>
                    </div>
                    <div>
                        <div class="text-2xl font-bold text-gray-900">{{ $pagination['current_page'] ?? 1 }}</div>
                        <div class="text-sm text-gray-500">Current Page</div>
                    </div>
                    <div>
                        <div class="text-2xl font-bold text-gray-900">{{ $pagination['last_page'] ?? 1 }}</div>
                        <div class="text-sm text-gray-500">Total Pages</div>
                    </div>
                </div>
            </div>

            <!-- Log Entries List -->
            <div class="space-y-2 max-h-96 overflow-y-auto">
                @foreach($logs as $log)
                    <div class="p-4 border border-gray-200 rounded-lg hover:bg-gray-50">
                        <div class="flex items-start justify-between">
                            <div class="flex-1">
                                <div class="flex items-center space-x-2 mb-2">
                                    <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium
                                        @if($log['level'] === 'ERROR' || $log['level'] === 'CRITICAL' || $log['level'] === 'EMERGENCY') bg-red-100 text-red-800
                                        @elseif($log['level'] === 'WARNING' || $log['level'] === 'ALERT') bg-yellow-100 text-yellow-800
                                        @elseif($log['level'] === 'INFO' || $log['level'] === 'NOTICE') bg-blue-100 text-blue-800
                                        @else bg-gray-100 text-gray-800
                                        @endif">
                                        {{ $log['level'] }}
                                    </span>
                                    @if($log['timestamp'])
                                        <span class="text-xs text-gray-500 font-mono">{{ $log['timestamp'] }}</span>
                                    @endif
                                    @if($log['environment'])
                                        <span class="text-xs text-gray-500 bg-gray-100 px-2 py-1 rounded">{{ $log['environment'] }}</span>
                                    @endif
                                </div>
                                <div class="text-sm text-gray-900 font-mono whitespace-pre-wrap">{!! $log['message'] !!}</div>
                            </div>
                            <div class="text-xs text-gray-400 ml-4 flex-shrink-0">
                                Line {{ $log['line_number'] }}
                            </div>
                        </div>
                    </div>
                @endforeach
            </div>

            <!-- Pagination -->
            @if($pagination && $pagination['last_page'] > 1)
                <div class="mt-6 flex items-center justify-between">
                    <div class="text-sm text-gray-700">
                        Showing {{ $pagination['from'] }} to {{ $pagination['to'] }} of {{ $pagination['total'] }} entries
                    </div>
                    <div class="flex space-x-2">
                        @if($pagination['current_page'] > 1)
                            <flux:button href="{{ request()->fullUrlWithQuery(['page' => $pagination['current_page'] - 1]) }}" variant="outline" size="sm">
                                <flux:icon name="chevron-left" class="w-4 h-4 mr-1" />
                                Previous
                            </flux:button>
                        @endif
                        
                        <!-- Page Numbers -->
                        @php
                            $start = max(1, $pagination['current_page'] - 2);
                            $end = min($pagination['last_page'], $pagination['current_page'] + 2);
                        @endphp
                        
                        @for($i = $start; $i <= $end; $i++)
                            <flux:button 
                                href="{{ request()->fullUrlWithQuery(['page' => $i]) }}" 
                                variant="{{ $i === $pagination['current_page'] ? 'primary' : 'outline' }}" 
                                size="sm"
                            >
                                {{ $i }}
                            </flux:button>
                        @endfor
                        
                        @if($pagination['has_more_pages'])
                            <flux:button href="{{ request()->fullUrlWithQuery(['page' => $pagination['current_page'] + 1]) }}" variant="outline" size="sm">
                                Next
                                <flux:icon name="chevron-right" class="w-4 h-4 ml-1" />
                            </flux:button>
                        @endif
                    </div>
                </div>
            @endif
        @else
            <div class="text-center py-12 text-gray-500">
                <flux:icon name="document-text" class="w-12 h-12 mx-auto mb-4 text-gray-300" />
                <p class="text-lg font-medium">No log entries found</p>
                <p class="text-sm">This log file appears to be empty or no entries match your filters.</p>
            </div>
        @endif
    </flux:card>

    <!-- Raw Log Content (Collapsible) -->
    <flux:card>
        <flux:heading size="lg" class="mb-4">Raw Log Content</flux:heading>
        <div class="bg-gray-900 text-green-400 p-4 rounded-lg overflow-x-auto">
            <pre class="text-xs font-mono whitespace-pre-wrap">{!! htmlspecialchars($logs->pluck('raw')->join("\n"), ENT_QUOTES, 'UTF-8') !!}</pre>
        </div>
    </flux:card>
</div>

<script>
// Auto-refresh functionality
let autoRefresh = false;
let refreshInterval;

function toggleAutoRefresh() {
    autoRefresh = !autoRefresh;
    const button = document.getElementById('auto-refresh-btn');
    
    if (autoRefresh) {
        button.textContent = 'Stop Auto Refresh';
        button.classList.remove('bg-blue-600', 'hover:bg-blue-700');
        button.classList.add('bg-red-600', 'hover:bg-red-700');
        
        refreshInterval = setInterval(function() {
            window.location.reload();
        }, 30000); // Refresh every 30 seconds
    } else {
        button.textContent = 'Auto Refresh';
        button.classList.remove('bg-red-600', 'hover:bg-red-700');
        button.classList.add('bg-blue-600', 'hover:bg-blue-700');
        
        clearInterval(refreshInterval);
    }
}

// Add auto-refresh button
document.addEventListener('DOMContentLoaded', function() {
    const header = document.querySelector('h1').parentElement;
    if (header && header.querySelector('div:last-child')) {
        const autoRefreshBtn = document.createElement('button');
        autoRefreshBtn.id = 'auto-refresh-btn';
        autoRefreshBtn.className = 'text-xs bg-blue-600 hover:bg-blue-700 text-white px-3 py-2 rounded';
        autoRefreshBtn.innerHTML = '🔄 Auto Refresh';
        autoRefreshBtn.onclick = toggleAutoRefresh;
        
        header.querySelector('div:last-child').appendChild(autoRefreshBtn);
    }
});
</script>
@endsection
