@extends('components.layouts.admin')

@section('title', 'KYC Management')

@section('content')
    <div class="space-y-6">
        <!-- Header -->
        <div class="flex justify-between items-center">
            <div>
                <h1 class="text-2xl font-bold text-gray-900">KYC Management</h1>
                <p class="text-gray-600">Review and manage KYC documents and user verification</p>
            </div>
            <div class="flex space-x-3">
                <flux:button icon="chart-bar" href="{{ route('admin.kyc.statistics') }}" variant="outline">
                    Statistics
                </flux:button>
            </div>
        </div>

        <!-- Statistics Cards -->
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
            <div class="bg-white rounded-xl border border-gray-200 shadow-sm p-6">
                <div class="flex items-center">
                    <div class="flex-shrink-0">
                        <div class="w-8 h-8 bg-blue-100 rounded-lg flex items-center justify-center">
                            <flux:icon name="document-text" class="w-5 h-5 text-blue-600" />
                        </div>
                    </div>
                    <div class="ml-4">
                        <p class="text-sm font-medium text-gray-500">Total Documents</p>
                        <p class="text-2xl font-bold text-gray-900">{{ number_format($stats['total_documents']) }}</p>
                    </div>
                </div>
            </div>

            <div class="bg-white rounded-xl border border-gray-200 shadow-sm p-6">
                <div class="flex items-center">
                    <div class="flex-shrink-0">
                        <div class="w-8 h-8 bg-yellow-100 rounded-lg flex items-center justify-center">
                            <flux:icon name="clock" class="w-5 h-5 text-yellow-600" />
                        </div>
                    </div>
                    <div class="ml-4">
                        <p class="text-sm font-medium text-gray-500">Pending Review</p>
                        <p class="text-2xl font-bold text-gray-900">{{ number_format($stats['pending_documents']) }}</p>
                    </div>
                </div>
            </div>

            <div class="bg-white rounded-xl border border-gray-200 shadow-sm p-6">
                <div class="flex items-center">
                    <div class="flex-shrink-0">
                        <div class="w-8 h-8 bg-green-100 rounded-lg flex items-center justify-center">
                            <flux:icon name="check-circle" class="w-5 h-5 text-green-600" />
                        </div>
                    </div>
                    <div class="ml-4">
                        <p class="text-sm font-medium text-gray-500">Approved</p>
                        <p class="text-2xl font-bold text-gray-900">{{ number_format($stats['approved_documents']) }}</p>
                    </div>
                </div>
            </div>

            <div class="bg-white rounded-xl border border-gray-200 shadow-sm p-6">
                <div class="flex items-center">
                    <div class="flex-shrink-0">
                        <div class="w-8 h-8 bg-red-100 rounded-lg flex items-center justify-center">
                            <flux:icon name="x-circle" class="w-5 h-5 text-red-600" />
                        </div>
                    </div>
                    <div class="ml-4">
                        <p class="text-sm font-medium text-gray-500">Rejected</p>
                        <p class="text-2xl font-bold text-gray-900">{{ number_format($stats['rejected_documents']) }}</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Filters -->
        <div class="bg-white rounded-xl border border-gray-200 shadow-sm p-6">
            <form method="GET" class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-4">
                <div>
                    <flux:label for="search">Search</flux:label>
                    <flux:input 
                        id="search" 
                        name="search" 
                        value="{{ request('search') }}" 
                        placeholder="Search by user name or email..."
                    />
                </div>

                <div>
                    <flux:label for="document_type">Document Type</flux:label>
                    <flux:select id="document_type" name="document_type">
                        <option value="">All Types</option>
                        @foreach(\App\Models\KycDocument::DOCUMENT_TYPES as $value => $label)
                            <option value="{{ $value }}" {{ request('document_type') == $value ? 'selected' : '' }}>
                                {{ $label }}
                            </option>
                        @endforeach
                    </flux:select>
                </div>

                <div>
                    <flux:label for="status">Status</flux:label>
                    <flux:select id="status" name="status">
                        <option value="">All Statuses</option>
                        @foreach(\App\Models\KycDocument::STATUSES as $value => $label)
                            <option value="{{ $value }}" {{ request('status') == $value ? 'selected' : '' }}>
                                {{ $label }}
                            </option>
                        @endforeach
                    </flux:select>
                </div>

                <div>
                    <flux:label for="verification_level">Verification Level</flux:label>
                    <flux:select id="verification_level" name="verification_level">
                        <option value="">All Levels</option>
                        @foreach(\App\Models\KycDocument::VERIFICATION_LEVELS as $value => $label)
                            <option value="{{ $value }}" {{ request('verification_level') == $value ? 'selected' : '' }}>
                                {{ $label }}
                            </option>
                        @endforeach
                    </flux:select>
                </div>

                <div class="flex items-end">
                    <flux:button type="submit" variant="primary" class="w-full">
                        Apply Filters
                    </flux:button>
                </div>
            </form>
        </div>

        <!-- Bulk Actions -->
        @if($documents->count() > 0)
            <div class="bg-white rounded-xl border border-gray-200 shadow-sm p-4">
                <form id="bulkForm" method="POST" action="{{ route('admin.kyc.bulk-approve') }}">
                    @csrf
                    <div class="flex items-center justify-between">
                        <div class="flex items-center space-x-4">
                            <flux:button type="button" onclick="selectAll()" variant="outline" size="sm">
                                Select All
                            </flux:button>
                            <flux:button type="button" onclick="deselectAll()" variant="outline" size="sm">
                                Deselect All
                            </flux:button>
                            <span class="text-sm text-gray-500" id="selectedCount">0 selected</span>
                        </div>
                        <div class="flex items-center space-x-3">
                            <flux:select name="verification_level" class="w-40">
                                <option value="basic">Basic</option>
                                <option value="enhanced">Enhanced</option>
                                <option value="full">Full</option>
                            </flux:select>
                            <flux:button type="submit" variant="primary" size="sm">
                                Bulk Approve
                            </flux:button>
                        </div>
                    </div>
                </form>
            </div>
        @endif

        <!-- Documents Table -->
        <div class="bg-white rounded-xl border border-gray-200 shadow-sm overflow-hidden">
            @if($documents->count() > 0)
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                    <input type="checkbox" id="selectAllCheckbox" onchange="toggleSelectAll()" class="rounded border-gray-300">
                                </th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                    User
                                </th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                    Document
                                </th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                    Status
                                </th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                    Verification Level
                                </th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                    Uploaded
                                </th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                    Reviewed By
                                </th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                    Actions
                                </th>
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-200">
                            @foreach($documents as $document)
                                <tr class="hover:bg-gray-50">
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <input type="checkbox" name="document_ids[]" value="{{ $document->id }}" class="document-checkbox rounded border-gray-300">
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="flex items-center">
                                            <div class="flex-shrink-0 h-10 w-10">
                                                <div class="h-10 w-10 rounded-full bg-gray-200 flex items-center justify-center">
                                                    <flux:icon name="user" class="w-5 h-5 text-gray-500" />
                                                </div>
                                            </div>
                                            <div class="ml-4">
                                                <div class="text-sm font-medium text-gray-900">{{ $document->user->name }}</div>
                                                <div class="text-sm text-gray-500">{{ $document->user->email }}</div>
                                            </div>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="text-sm font-medium text-gray-900">{{ $document->document_type_label }}</div>
                                        <div class="text-sm text-gray-500">{{ $document->document_name }}</div>
                                        <div class="text-xs text-gray-400">{{ $document->file_size_human }}</div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full {{ 
                                            $document->status === 'approved' ? 'bg-green-100 text-green-800' : 
                                            ($document->status === 'rejected' ? 'bg-red-100 text-red-800' : 
                                            ($document->status === 'expired' ? 'bg-gray-100 text-gray-800' : 'bg-yellow-100 text-yellow-800')) 
                                        }}">
                                            {{ $document->status_label }}
                                        </span>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                        {{ $document->verification_level_label ?? 'N/A' }}
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                        {{ $document->created_at->format('M d, Y') }}
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                        @if($document->reviewer)
                                            <div>{{ $document->reviewer->name }}</div>
                                            <div class="text-xs text-gray-400">{{ $document->reviewed_at?->format('M d, Y') }}</div>
                                        @else
                                            <span class="text-gray-400">Not reviewed</span>
                                        @endif
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                        <div class="flex space-x-2">
                                            <flux:button 
                                                href="{{ route('admin.kyc.show', $document) }}" 
                                                variant="outline" 
                                                size="sm"
                                            >
                                                View
                                            </flux:button>
                                            
                                            @if($document->status === 'pending')
                                                <form method="POST" action="{{ route('admin.kyc.approve', $document) }}" class="inline" onsubmit="return confirm('Approve this document?');">
                                                    @csrf
                                                    <flux:button 
                                                        type="submit"
                                                        variant="primary" 
                                                        size="sm"
                                                    >
                                                        Approve
                                                    </flux:button>
                                                </form>
                                                
                                                <form method="POST" action="{{ route('admin.kyc.reject', $document) }}" class="inline" onsubmit="return confirm('Reject this document?');">
                                                    @csrf
                                                    <flux:button 
                                                        type="submit"
                                                        variant="danger" 
                                                        size="sm"
                                                    >
                                                        Reject
                                                    </flux:button>
                                                </form>
                                            @endif
                                            
                                            <flux:button 
                                                href="{{ route('admin.kyc.download', $document) }}" 
                                                variant="outline" 
                                                size="sm"
                                            >
                                                Download
                                            </flux:button>
                                        </div>
                                    </td>
                                </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>

                <!-- Pagination -->
                <div class="bg-white px-4 py-3 border-t border-gray-200 sm:px-6">
                    {{ $documents->links() }}
                </div>
            @else
                <div class="text-center py-12">
                    <flux:icon name="document-text" class="mx-auto h-12 w-12 text-gray-400" />
                    <h3 class="mt-2 text-sm font-medium text-gray-900">No KYC documents found</h3>
                    <p class="mt-1 text-sm text-gray-500">No documents match your current filters.</p>
                </div>
            @endif
        </div>
    </div>

    <script>
        function selectAll() {
            const checkboxes = document.querySelectorAll('.document-checkbox');
            checkboxes.forEach(checkbox => checkbox.checked = true);
            updateSelectedCount();
        }

        function deselectAll() {
            const checkboxes = document.querySelectorAll('.document-checkbox');
            checkboxes.forEach(checkbox => checkbox.checked = false);
            updateSelectedCount();
        }

        function toggleSelectAll() {
            const selectAllCheckbox = document.getElementById('selectAllCheckbox');
            const checkboxes = document.querySelectorAll('.document-checkbox');
            checkboxes.forEach(checkbox => checkbox.checked = selectAllCheckbox.checked);
            updateSelectedCount();
        }

        function updateSelectedCount() {
            const checkboxes = document.querySelectorAll('.document-checkbox:checked');
            document.getElementById('selectedCount').textContent = `${checkboxes.length} selected`;
        }

        // Update count when individual checkboxes change
        document.addEventListener('change', function(e) {
            if (e.target.classList.contains('document-checkbox')) {
                updateSelectedCount();
            }
        });

        // Update select all checkbox state
        document.addEventListener('change', function(e) {
            if (e.target.classList.contains('document-checkbox')) {
                const checkboxes = document.querySelectorAll('.document-checkbox');
                const checkedBoxes = document.querySelectorAll('.document-checkbox:checked');
                const selectAllCheckbox = document.getElementById('selectAllCheckbox');
                
                if (checkedBoxes.length === 0) {
                    selectAllCheckbox.indeterminate = false;
                    selectAllCheckbox.checked = false;
                } else if (checkedBoxes.length === checkboxes.length) {
                    selectAllCheckbox.indeterminate = false;
                    selectAllCheckbox.checked = true;
                } else {
                    selectAllCheckbox.indeterminate = true;
                }
            }
        });
    </script>
@endsection
