@extends('components.layouts.admin')

@section('title', 'Compliance Management')

@section('content')
<div class="space-y-6">
    <!-- Success Message -->
    @if (session('success'))
        <x-admin.message type="success">
            {{ session('success') }}
        </x-admin.message>
    @endif

    <!-- Error Message -->
    @if (session('error'))
        <x-admin.message type="error">
            {{ session('error') }}
        </x-admin.message>
    @endif

    <!-- Header -->
    <div class="flex flex-col sm:flex-row sm:justify-between sm:items-center space-y-4 sm:space-y-0">
        <div>
            <h1 class="text-2xl font-bold text-gray-900">Compliance Management</h1>
            <p class="text-gray-600">Monitor and manage compliance logs and risk assessments</p>
        </div>
        <div class="flex flex-wrap gap-2 sm:gap-3">
            <flux:button href="{{ route('admin.compliance.statistics') }}" variant="outline" class="w-full sm:w-auto">
                <flux:icon name="chart-bar" class="w-4 h-4 mr-2" />
                Statistics
            </flux:button>
        </div>
    </div>

    <!-- AML Monitoring Actions -->
    <flux:card>
        <div class="flex flex-col lg:flex-row lg:items-center lg:justify-between space-y-4 lg:space-y-0">
            <div class="flex-1">
                <h3 class="text-lg font-semibold text-gray-900">AML Monitoring</h3>
                <p class="text-sm text-gray-600">Run automated AML monitoring on recent transactions and users</p>
            </div>
            <div class="flex flex-col sm:flex-row gap-2 sm:gap-3 lg:flex-shrink-0">
                <flux:button 
                    onclick="runAmlMonitoring()" 
                    variant="primary" 
                    id="aml-monitor-btn"
                    class="flex items-center justify-center w-full sm:w-auto"
                >
                    <flux:icon name="shield-check" class="w-4 h-4 mr-2" />
                    <span class="hidden sm:inline">Run AML Monitoring</span>
                    <span class="sm:hidden">Run Monitoring</span>
                </flux:button>
                <flux:button 
                    onclick="runAmlMonitoring(true)" 
                    variant="outline" 
                    id="aml-monitor-force-btn"
                    class="flex items-center justify-center w-full sm:w-auto"
                >
                    <flux:icon name="exclamation-triangle" class="w-4 h-4 mr-2" />
                    <span class="hidden sm:inline">Force Run</span>
                    <span class="sm:hidden">Force</span>
                </flux:button>
            </div>
        </div>
        
        <!-- AML Monitoring Results -->
        <div id="aml-results" class="mt-4 hidden">
            <div class="bg-gray-50 rounded-lg p-4">
                <h4 class="text-sm font-medium text-gray-900 mb-2">AML Monitoring Results</h4>
                <div id="aml-results-content" class="text-sm text-gray-600"></div>
            </div>
        </div>
    </flux:card>

    <!-- Filters -->
    <flux:card>
        <form method="GET" class="space-y-4">
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-4">
                <!-- Search -->
                <div>
                    <flux:label for="search">Search</flux:label>
                    <flux:input 
                        id="search" 
                        name="search" 
                        value="{{ request('search') }}" 
                        placeholder="User name or email..."
                    />
                </div>

                <!-- Status Filter -->
                <div>
                    <flux:label for="status">Status</flux:label>
                    <flux:select id="status" name="status">
                        <option value="">All Statuses</option>
                        @foreach($statuses as $status)
                            <option value="{{ $status }}" {{ request('status') == $status ? 'selected' : '' }}>
                                {{ ucfirst(str_replace('_', ' ', $status)) }}
                            </option>
                        @endforeach
                    </flux:select>
                </div>

                <!-- Severity Filter -->
                <div>
                    <flux:label for="severity">Severity</flux:label>
                    <flux:select id="severity" name="severity">
                        <option value="">All Severities</option>
                        @foreach($severities as $severity)
                            <option value="{{ $severity }}" {{ request('severity') == $severity ? 'selected' : '' }}>
                                {{ ucfirst($severity) }}
                            </option>
                        @endforeach
                    </flux:select>
                </div>

                <!-- Type Filter -->
                <div>
                    <flux:label for="type">Type</flux:label>
                    <flux:select id="type" name="type">
                        <option value="">All Types</option>
                        @foreach($types as $type)
                            <option value="{{ $type }}" {{ request('type') == $type ? 'selected' : '' }}>
                                {{ ucfirst(str_replace('_', ' ', $type)) }}
                            </option>
                        @endforeach
                    </flux:select>
                </div>

                <!-- Date From -->
                <div>
                    <flux:label for="date_from">Date From</flux:label>
                    <flux:date-picker 
                        id="date_from" 
                        name="date_from" 
                        value="{{ request('date_from') }}"
                        selectable-header
                        class="text-xs"
                    />
                </div>
            </div>

            <div class="flex justify-between items-center">
                <div class="flex space-x-2">
                    <flux:button icon="magnifying-glass" type="submit" variant="primary" class="text-xs">
                        Filter
                    </flux:button>
                    <flux:button icon="x-mark" href="{{ route('admin.compliance.index') }}" variant="outline" class="text-xs">
                        Clear
                    </flux:button>
                </div>

                <!-- Bulk Actions -->
                @if($logs->count() > 0)
                    <div class="flex items-center space-x-2">
                        <flux:button 
                            type="button" 
                            variant="outline" 
                            onclick="toggleBulkActions()"
                            class="text-xs"
                            icon="check"
                        >
                            Bulk Actions
                        </flux:button>
                    </div>
                @endif
            </div>
        </form>
    </flux:card>

    <!-- Bulk Actions Form -->
    <div id="bulk-actions" class="hidden">
        <flux:card>
            <form method="POST" action="{{ route('admin.compliance.bulk-approve') }}" onsubmit="return confirm('Approve selected compliance logs?');">
                @csrf
                <div class="flex items-center space-x-4">
                    <div class="flex-1">
                        <flux:label for="bulk_resolution_notes">Resolution Notes</flux:label>
                        <flux:textarea 
                            id="bulk_resolution_notes" 
                            name="resolution_notes" 
                            placeholder="Enter resolution notes for bulk approval..."
                            rows="2"
                        />
                    </div>
                    <div class="flex space-x-2">
                        <flux:button icon="check" type="submit" variant="primary" class="text-xs">
                            Approve Selected
                        </flux:button>
                        <flux:button icon="x-mark" type="button" variant="outline" onclick="toggleBulkActions()" class="text-xs">
                            Cancel
                        </flux:button>
                    </div>
                </div>
            </form>
        </flux:card>
    </div>

    <!-- Compliance Logs Table -->
    <flux:card>
        @if($logs->count() > 0)
            <div class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-200">
                    <thead class="bg-gray-50">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                <input type="checkbox" id="select-all" onchange="toggleAllCheckboxes()" class="rounded">
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Type
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                User
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Description
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Severity
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Status
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Created
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Actions
                            </th>
                        </tr>
                    </thead>
                    <tbody class="bg-white divide-y divide-gray-200">
                        @foreach($logs as $log)
                            <tr class="hover:bg-gray-50">
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <input type="checkbox" name="log_ids[]" value="{{ $log->id }}" class="log-checkbox rounded">
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="text-sm font-medium text-gray-900">
                                        {{ $log->type_label }}
                                    </div>
                                    <div class="text-sm text-gray-500">
                                        {{ $log->category_label }}
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    @if($log->user)
                                        <div class="text-sm font-medium text-gray-900">
                                            {{ $log->user->name }}
                                        </div>
                                        <div class="text-sm text-gray-500">
                                            {{ $log->user->email }}
                                        </div>
                                    @else
                                        <span class="text-sm text-gray-500">N/A</span>
                                    @endif
                                </td>
                                <td class="px-6 py-4">
                                    <div class="text-sm text-gray-900 max-w-xs truncate">
                                        {{ $log->description }}
                                    </div>
                                    @if($log->amount)
                                        <div class="text-sm text-gray-500">
                                            {{ number_format($log->amount, 2) }} {{ $log->currency }}
                                        </div>
                                    @endif
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium {{ $log->severity_color }}">
                                        {{ ucfirst($log->severity) }}
                                    </span>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium {{ $log->status_color }}">
                                        {{ ucfirst(str_replace('_', ' ', $log->status)) }}
                                    </span>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                    {{ $log->created_at->format('M j, Y g:i A') }}
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm font-medium space-x-2">
                                    <flux:button 
                                        href="{{ route('admin.compliance.show', $log) }}" 
                                        variant="outline" 
                                        size="sm"
                                    >
                                        View
                                    </flux:button>
                                    
                                    @if($log->status === 'pending' || $log->status === 'under_review')
                                        <form method="POST" action="{{ route('admin.compliance.approve', $log) }}" class="inline" onsubmit="return confirm('Approve this compliance log?');">
                                            @csrf
                                            <flux:button icon="check" type="submit" variant="primary" size="sm">
                                                Approve
                                            </flux:button>
                                        </form>
                                        
                                        <form method="POST" action="{{ route('admin.compliance.reject', $log) }}" class="inline" onsubmit="return confirm('Reject this compliance log?');">
                                            @csrf
                                            <flux:button type="submit" variant="danger" size="sm">
                                                Reject
                                            </flux:button>
                                        </form>
                                    @endif
                                </td>
                            </tr>
                        @endforeach
                    </tbody>
                </table>
            </div>

            <!-- Pagination -->
            <div class="px-6 py-4 border-t border-gray-200">
                {{ $logs->links() }}
            </div>
        @else
            <div class="text-center py-12">
                <flux:icon name="exclamation-triangle" class="w-12 h-12 text-gray-400 mx-auto mb-4" />
                <h3 class="text-lg font-medium text-gray-900 mb-2">No compliance logs found</h3>
                <p class="text-gray-500">No compliance logs match your current filters.</p>
            </div>
        @endif
    </flux:card>
</div>

<script>
// Set up global variables
const amlMonitorUrl = '{{ route("admin.compliance.aml-monitor") }}';
const csrfToken = document.querySelector('meta[name="csrf-token"]').getAttribute('content');

function toggleBulkActions() {
    const bulkActions = document.getElementById('bulk-actions');
    bulkActions.classList.toggle('hidden');
}

function toggleAllCheckboxes() {
    const selectAll = document.getElementById('select-all');
    const checkboxes = document.querySelectorAll('.log-checkbox');
    
    checkboxes.forEach(checkbox => {
        checkbox.checked = selectAll.checked;
    });
}

// Update select-all checkbox when individual checkboxes change
document.querySelectorAll('.log-checkbox').forEach(checkbox => {
    checkbox.addEventListener('change', function() {
        const checkboxes = document.querySelectorAll('.log-checkbox');
        const checkedBoxes = document.querySelectorAll('.log-checkbox:checked');
        const selectAll = document.getElementById('select-all');
        
        selectAll.checked = checkboxes.length === checkedBoxes.length;
        selectAll.indeterminate = checkedBoxes.length > 0 && checkedBoxes.length < checkboxes.length;
    });
});

// AML Monitoring Functions
function runAmlMonitoring(force = false) {
    const btn = force ? document.getElementById('aml-monitor-force-btn') : document.getElementById('aml-monitor-btn');
    const resultsDiv = document.getElementById('aml-results');
    const resultsContent = document.getElementById('aml-results-content');
    
    // Disable button and show loading
    btn.disabled = true;
    btn.innerHTML = '<svg class="w-4 h-4 mr-2 animate-spin" fill="none" viewBox="0 0 24 24"><circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle><path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path></svg>Running...';
    
    // Show results area
    resultsDiv.classList.remove('hidden');
    resultsContent.innerHTML = '<div class="flex items-center"><svg class="w-4 h-4 mr-2 animate-spin" fill="none" viewBox="0 0 24 24"><circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle><path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path></svg>Running AML monitoring...</div>';
    
    // Make AJAX request
    fetch(amlMonitorUrl, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': csrfToken
        },
        body: JSON.stringify({
            force: force
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            resultsContent.innerHTML = `
                <div class="space-y-3">
                    <div class="grid grid-cols-2 gap-4">
                        <div class="bg-white p-3 rounded border">
                            <div class="text-xs text-gray-500">Transactions Monitored</div>
                            <div class="text-lg font-semibold text-gray-900">${data.results.transactions_monitored}</div>
                        </div>
                        <div class="bg-white p-3 rounded border">
                            <div class="text-xs text-gray-500">Transactions Flagged</div>
                            <div class="text-lg font-semibold text-red-600">${data.results.transactions_flagged}</div>
                        </div>
                        <div class="bg-white p-3 rounded border">
                            <div class="text-xs text-gray-500">Users Monitored</div>
                            <div class="text-lg font-semibold text-gray-900">${data.results.users_monitored}</div>
                        </div>
                        <div class="bg-white p-3 rounded border">
                            <div class="text-xs text-gray-500">Users Flagged</div>
                            <div class="text-lg font-semibold text-red-600">${data.results.users_flagged}</div>
                        </div>
                    </div>
                    <div class="mt-4 p-3 bg-blue-50 rounded border-l-4 border-blue-400">
                        <div class="text-sm text-blue-800">
                            <strong>Suspicious Activity Threshold:</strong> $${data.stats.suspicious_threshold.toLocaleString()}
                        </div>
                    </div>
                </div>
            `;
        } else {
            resultsContent.innerHTML = `<div class="text-red-600">Error: ${data.message}</div>`;
        }
    })
    .catch(error => {
        console.error('Error:', error);
        resultsContent.innerHTML = `<div class="text-red-600">Error running AML monitoring: ${error.message}</div>`;
    })
    .finally(() => {
        // Re-enable button
        btn.disabled = false;
        if (force) {
            btn.innerHTML = '<svg class="w-4 h-4 mr-2" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L3.732 16.5c-.77.833.192 2.5 1.732 2.5z" /></svg>Force Run';
        } else {
            btn.innerHTML = '<svg class="w-4 h-4 mr-2" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.03 9-11.622 0-1.042-.133-2.052-.382-3.016z" /></svg>Run AML Monitoring';
        }
    });
}
</script>
@endsection
