@extends('components.layouts.admin')

@section('title', 'Account Analytics')

@section('content')
<div class="space-y-6">
    <!-- Success Message -->
    @if (session('success'))
        <x-admin.message type="success">
            {{ session('success') }}
        </x-admin.message>
    @endif

    <!-- Error Message -->
    @if (session('error'))
        <x-admin.message type="error">
            {{ session('error') }}
        </x-admin.message>
    @endif

    <!-- Header -->
    <div class="flex justify-between items-center">
        <div>
            <h1 class="text-2xl font-bold text-gray-900">Account Analytics</h1>
            <p class="text-gray-600">Account creation trends, performance metrics, and distribution insights</p>
        </div>
        <div class="flex space-x-3">
            <flux:select id="period-select" onchange="updatePeriod(this.value)">
                <option value="7d" {{ $period === '7d' ? 'selected' : '' }}>Last 7 Days</option>
                <option value="30d" {{ $period === '30d' ? 'selected' : '' }}>Last 30 Days</option>
                <option value="90d" {{ $period === '90d' ? 'selected' : '' }}>Last 90 Days</option>
                <option value="1y" {{ $period === '1y' ? 'selected' : '' }}>Last Year</option>
            </flux:select>
            <flux:button href="{{ route('admin.analytics.index') }}" variant="outline" class="text-xs">
                <flux:icon name="arrow-left" class="w-4 h-4 mr-2" />
                Back to Analytics
            </flux:button>
        </div>
    </div>

    <!-- Account Summary Cards -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <flux:card>
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <flux:icon name="credit-card" class="w-8 h-8 text-blue-600" />
                </div>
                <div class="ml-4">
                    <p class="text-sm font-medium text-gray-500">Total Accounts</p>
                    <p class="text-2xl font-semibold text-gray-900">{{ number_format($accountPerformance['total_accounts']) }}</p>
                </div>
            </div>
        </flux:card>

        <flux:card>
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <flux:icon name="check-circle" class="w-8 h-8 text-green-600" />
                </div>
                <div class="ml-4">
                    <p class="text-sm font-medium text-gray-500">Active Accounts</p>
                    <p class="text-2xl font-semibold text-gray-900">{{ number_format($accountPerformance['active_accounts']) }}</p>
                </div>
            </div>
        </flux:card>

        <flux:card>
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <flux:icon name="exclamation-triangle" class="w-8 h-8 text-yellow-600" />
                </div>
                <div class="ml-4">
                    <p class="text-sm font-medium text-gray-500">Suspended</p>
                    <p class="text-2xl font-semibold text-gray-900">{{ number_format($accountPerformance['suspended_accounts']) }}</p>
                </div>
            </div>
        </flux:card>

        <flux:card>
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <flux:icon name="x-circle" class="w-8 h-8 text-red-600" />
                </div>
                <div class="ml-4">
                    <p class="text-sm font-medium text-gray-500">Closed</p>
                    <p class="text-2xl font-semibold text-gray-900">{{ number_format($accountPerformance['closed_accounts']) }}</p>
                </div>
            </div>
        </flux:card>
    </div>

    <!-- Financial Summary -->
    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
        <flux:card>
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-sm font-medium text-gray-500">Total Balance</p>
                    <p class="text-3xl font-bold text-gray-900">${{ number_format($accountPerformance['total_balance'], 2) }}</p>
                </div>
                <flux:icon name="currency-dollar" class="w-12 h-12 text-green-600" />
            </div>
        </flux:card>

        <flux:card>
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-sm font-medium text-gray-500">Average Balance</p>
                    <p class="text-3xl font-bold text-gray-900">${{ number_format($accountPerformance['average_balance'], 2) }}</p>
                </div>
                <flux:icon name="calculator" class="w-12 h-12 text-blue-600" />
            </div>
        </flux:card>
    </div>

    <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <!-- Account Creation Trend -->
        <flux:card>
            <flux:heading size="lg" class="mb-4">Account Creation Trend</flux:heading>
            @if($accountCreation->count() > 0)
                <div class="space-y-3">
                    @foreach($accountCreation as $data)
                        <div class="flex justify-between items-center">
                            <span class="text-sm font-medium text-gray-900">
                                {{ \Carbon\Carbon::parse($data->date)->format('M j, Y') }}
                            </span>
                            <div class="flex items-center space-x-2">
                                <div class="w-32 bg-gray-200 rounded-full h-2">
                                    <div class="bg-blue-600 h-2 rounded-full" style="width: {{ ($data->count / $accountCreation->max('count')) * 100 }}%"></div>
                                </div>
                                <span class="text-sm text-gray-500 w-12 text-right">{{ $data->count }}</span>
                            </div>
                        </div>
                    @endforeach
                </div>
            @else
                <p class="text-sm text-gray-500">No account creation data available for the selected period.</p>
            @endif
        </flux:card>

        <!-- Account Status Distribution -->
        <flux:card>
            <flux:heading size="lg" class="mb-4">Account Status Distribution</flux:heading>
            @if($accountPerformance['by_status']->count() > 0)
                <div class="space-y-3">
                    @foreach($accountPerformance['by_status'] as $status => $count)
                        <div class="flex justify-between items-center">
                            <span class="text-sm font-medium text-gray-900">
                                {{ ucfirst(str_replace('_', ' ', $status)) }}
                            </span>
                            <div class="flex items-center space-x-2">
                                <div class="w-32 bg-gray-200 rounded-full h-2">
                                    <div class="h-2 rounded-full {{ 
                                        $status === 'active' ? 'bg-green-600' : 
                                        ($status === 'suspended' ? 'bg-yellow-600' : 
                                        ($status === 'closed' ? 'bg-red-600' : 'bg-gray-600')) 
                                    }}" style="width: {{ ($count / $accountPerformance['by_status']->sum()) * 100 }}%"></div>
                                </div>
                                <span class="text-sm text-gray-500 w-12 text-right">{{ $count }}</span>
                            </div>
                        </div>
                    @endforeach
                </div>
            @else
                <p class="text-sm text-gray-500">No account status data available.</p>
            @endif
        </flux:card>
    </div>

    <!-- Account Type Analysis -->
    <flux:card>
        <flux:heading size="lg" class="mb-4">Account Type Distribution</flux:heading>
        @if($accountPerformance['by_type']->count() > 0)
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                @foreach($accountPerformance['by_type'] as $type => $count)
                    <div class="p-4 bg-gray-50 rounded-lg">
                        <div class="flex justify-between items-center mb-2">
                            <span class="text-sm font-medium text-gray-900">
                                {{ ucfirst(str_replace('_', ' ', $type)) }}
                            </span>
                            <span class="text-xs text-gray-500">{{ $count }} accounts</span>
                        </div>
                        <div class="text-2xl font-bold text-gray-900 mb-1">
                            {{ number_format(($count / $accountPerformance['total_accounts']) * 100, 1) }}%
                        </div>
                        <div class="text-sm text-gray-500">
                            of total accounts
                        </div>
                    </div>
                @endforeach
            </div>
        @else
            <p class="text-sm text-gray-500">No account type data available.</p>
        @endif
    </flux:card>

    <!-- Performance Metrics -->
    <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <!-- Account Health Metrics -->
        <flux:card>
            <flux:heading size="lg" class="mb-4">Account Health Metrics</flux:heading>
            <div class="space-y-4">
                <div class="flex justify-between items-center">
                    <span class="text-sm text-gray-700">Active Rate</span>
                    <span class="text-lg font-semibold text-gray-900">
                        {{ $accountPerformance['total_accounts'] > 0 ? number_format(($accountPerformance['active_accounts'] / $accountPerformance['total_accounts']) * 100, 1) : 0 }}%
                    </span>
                </div>
                <div class="flex justify-between items-center">
                    <span class="text-sm text-gray-700">Suspension Rate</span>
                    <span class="text-lg font-semibold text-gray-900">
                        {{ $accountPerformance['total_accounts'] > 0 ? number_format(($accountPerformance['suspended_accounts'] / $accountPerformance['total_accounts']) * 100, 1) : 0 }}%
                    </span>
                </div>
                <div class="flex justify-between items-center">
                    <span class="text-sm text-gray-700">Closure Rate</span>
                    <span class="text-lg font-semibold text-gray-900">
                        {{ $accountPerformance['total_accounts'] > 0 ? number_format(($accountPerformance['closed_accounts'] / $accountPerformance['total_accounts']) * 100, 1) : 0 }}%
                    </span>
                </div>
                <div class="flex justify-between items-center">
                    <span class="text-sm text-gray-700">Health Score</span>
                    <span class="text-lg font-semibold text-gray-900">
                        @php
                            $activeRate = $accountPerformance['total_accounts'] > 0 ? ($accountPerformance['active_accounts'] / $accountPerformance['total_accounts']) * 100 : 0;
                            $suspensionRate = $accountPerformance['total_accounts'] > 0 ? ($accountPerformance['suspended_accounts'] / $accountPerformance['total_accounts']) * 100 : 0;
                            $healthScore = max(0, $activeRate - ($suspensionRate * 0.5));
                        @endphp
                        {{ number_format($healthScore, 1) }}%
                    </span>
                </div>
            </div>
        </flux:card>

        <!-- Financial Metrics -->
        <flux:card>
            <flux:heading size="lg" class="mb-4">Financial Metrics</flux:heading>
            <div class="space-y-4">
                <div class="flex justify-between items-center">
                    <span class="text-sm text-gray-700">Total Balance</span>
                    <span class="text-lg font-semibold text-gray-900">${{ number_format($accountPerformance['total_balance'], 2) }}</span>
                </div>
                <div class="flex justify-between items-center">
                    <span class="text-sm text-gray-700">Average Balance</span>
                    <span class="text-lg font-semibold text-gray-900">${{ number_format($accountPerformance['average_balance'], 2) }}</span>
                </div>
                <div class="flex justify-between items-center">
                    <span class="text-sm text-gray-700">Balance per Active Account</span>
                    <span class="text-lg font-semibold text-gray-900">
                        ${{ $accountPerformance['active_accounts'] > 0 ? number_format($accountPerformance['total_balance'] / $accountPerformance['active_accounts'], 2) : '0.00' }}
                    </span>
                </div>
                <div class="flex justify-between items-center">
                    <span class="text-sm text-gray-700">High Value Accounts</span>
                    <span class="text-lg font-semibold text-gray-900">
                        @php
                            $highValueThreshold = $accountPerformance['average_balance'] * 2;
                            // This would need to be calculated from actual data
                            $highValueCount = 0; // Placeholder
                        @endphp
                        {{ $highValueCount }} accounts
                    </span>
                </div>
            </div>
        </flux:card>
    </div>

    <!-- Account Creation Analysis -->
    <flux:card>
        <flux:heading size="lg" class="mb-4">Account Creation Analysis</flux:heading>
        @if($accountCreation->count() > 0)
            <div class="space-y-4">
                <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <div class="text-center p-4 bg-gray-50 rounded-lg">
                        <div class="text-2xl font-bold text-gray-900">{{ $accountCreation->sum('count') }}</div>
                        <div class="text-sm text-gray-500">Total Created ({{ ucfirst($period) }})</div>
                    </div>
                    <div class="text-center p-4 bg-gray-50 rounded-lg">
                        <div class="text-2xl font-bold text-gray-900">
                            {{ number_format($accountCreation->avg('count'), 1) }}
                        </div>
                        <div class="text-sm text-gray-500">Daily Average</div>
                    </div>
                    <div class="text-center p-4 bg-gray-50 rounded-lg">
                        <div class="text-2xl font-bold text-gray-900">{{ $accountCreation->max('count') }}</div>
                        <div class="text-sm text-gray-500">Peak Day</div>
                    </div>
                </div>
                
                <div class="mt-6">
                    <h4 class="text-sm font-medium text-gray-700 mb-3">Creation Trend</h4>
                    <div class="space-y-2">
                        @foreach($accountCreation->take(10) as $data)
                            <div class="flex justify-between items-center">
                                <span class="text-sm text-gray-600">
                                    {{ \Carbon\Carbon::parse($data->date)->format('M j') }}
                                </span>
                                <div class="flex items-center space-x-2">
                                    <div class="w-24 bg-gray-200 rounded-full h-1">
                                        <div class="bg-blue-600 h-1 rounded-full" style="width: {{ ($data->count / $accountCreation->max('count')) * 100 }}%"></div>
                                    </div>
                                    <span class="text-xs text-gray-500 w-8 text-right">{{ $data->count }}</span>
                                </div>
                            </div>
                        @endforeach
                    </div>
                </div>
            </div>
        @else
            <p class="text-sm text-gray-500">No account creation data available for the selected period.</p>
        @endif
    </flux:card>
</div>

<script>
function updatePeriod(period) {
    const url = new URL(window.location);
    url.searchParams.set('period', period);
    window.location.href = url.toString();
}
</script>
@endsection
