<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('two_factor_auth', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->onDelete('cascade');
            
            // 2FA Configuration
            $table->boolean('enabled')->default(false);
            $table->enum('method', ['totp', 'email', 'authenticator_app', 'passkey'])->default('totp');
            $table->string('secret_key')->nullable(); // For TOTP/HOTP
            $table->string('phone_number')->nullable(); // For SMS
            $table->string('email')->nullable(); // For email verification
            
            // TOTP Specific
            $table->string('totp_secret')->nullable();
            $table->integer('totp_digits')->default(6);
            $table->integer('totp_period')->default(30); // seconds
            $table->string('totp_algorithm')->default('sha1'); // sha1, sha256, sha512
            $table->integer('totp_counter')->default(0); // For HOTP
            
            //Email Verification
            $table->string('verification_code')->nullable();
            $table->timestamp('verification_code_expires_at')->nullable();
            $table->integer('verification_attempts')->default(0);
            $table->timestamp('last_verification_attempt')->nullable();
            
            // Backup Codes
            $table->json('backup_codes')->nullable(); // Array of hashed backup codes
            $table->integer('backup_codes_remaining')->default(10);
            $table->timestamp('backup_codes_generated_at')->nullable();
            
            // Passkey Support (FIDO2/WebAuthn)
            $table->json('passkeys')->nullable(); // Array of registered passkeys
            $table->string('passkey_name')->nullable();
            $table->string('passkey_credential_id')->nullable(); // Unique credential identifier
            $table->string('passkey_public_key')->nullable(); // Public key for verification
            
            // Security Settings
            $table->boolean('remember_device')->default(false);
            $table->integer('remember_device_days')->default(30);
            $table->boolean('require_on_login')->default(true);
            $table->boolean('require_on_transfer')->default(true);
            $table->boolean('require_on_settings_change')->default(true);
            $table->decimal('transfer_threshold', 20, 4)->nullable(); // Amount above which 2FA is required
            
            // Device Management
            $table->json('trusted_devices')->nullable(); // Array of trusted device information
            $table->string('current_device_id')->nullable();
            $table->timestamp('device_verified_at')->nullable();
            
            // Recovery Options
            $table->boolean('recovery_email_enabled')->default(false);
            $table->string('recovery_email')->nullable();
            $table->boolean('recovery_phone_enabled')->default(false);
            $table->string('recovery_phone')->nullable();
            $table->text('recovery_questions')->nullable(); // JSON of security questions
            
            // Activity Tracking
            $table->timestamp('last_used_at')->nullable();
            $table->timestamp('last_failed_attempt')->nullable();
            $table->integer('failed_attempts')->default(0);
            $table->timestamp('locked_until')->nullable(); // Account lockout after failed attempts
            
            // Compliance and Audit
            $table->enum('compliance_status', ['compliant', 'non_compliant', 'under_review'])->default('compliant');
            $table->text('compliance_notes')->nullable();
            $table->timestamp('last_compliance_check')->nullable();
            
            // Admin Override
            $table->boolean('admin_override')->default(false);
            $table->foreignId('admin_override_by')->nullable()->constrained('users');
            $table->timestamp('admin_override_at')->nullable();
            $table->text('admin_override_reason')->nullable();
            
            // Timestamps
            $table->timestamps();
            $table->softDeletes();
            
            // Indexes
            $table->index(['user_id']);
            $table->index(['enabled']);
            $table->index(['method']);
            $table->index(['verification_code']);
            $table->index(['verification_code_expires_at']);
            $table->index(['locked_until']);
            $table->index(['compliance_status']);
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('two_factor_auth');
    }
};
