<?php

namespace App\Services;

use App\Models\Notification;
use App\Models\User;
use App\Mail\BankingNotificationMail;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;

class NotificationService
{
    /**
     * Create a new notification and optionally send email.
     */
    public function createNotification(
        User $user,
        string $type,
        string $title,
        string $message,
        array $data = [],
        bool $sendEmail = true
    ): Notification {
        $notification = Notification::create([
            'user_id' => $user->id,
            'type' => $type,
            'title' => $title,
            'message' => $message,
            'data' => $data,
        ]);

        // Send email if requested
        if ($sendEmail) {
            $this->sendEmailNotification($notification);
        }

        return $notification;
    }

    /**
     * Send email notification.
     */
    public function sendEmailNotification(Notification $notification): bool
    {
        try {
            Mail::to($notification->user->email)->send(
                new BankingNotificationMail($notification)
            );

            $notification->markEmailSent();
            return true;
        } catch (\Exception $e) {
            Log::error('Failed to send notification email: ' . $e->getMessage(), [
                'notification_id' => $notification->id,
                'user_id' => $notification->user_id,
                'type' => $notification->type,
            ]);
            return false;
        }
    }

    /**
     * Login IP address change notification.
     */
    public function notifyLoginIpChange(User $user, string $newIp, string $location = null): void
    {
        $this->createNotification(
            user: $user,
            type: 'login_ip_change',
            title: 'New Login Detected',
            message: "A new login was detected from IP address {$newIp}" . ($location ? " in {$location}" : '') . ". If this wasn't you, please secure your account immediately.",
            data: [
                'ip_address' => $newIp,
                'location' => $location,
                'timestamp' => now()->toISOString(),
            ]
        );
    }

    /**
     * New user welcome notification.
     */
    public function notifyNewUserWelcome(User $user): void
    {
        $this->createNotification(
            user: $user,
            type: 'new_user_welcome',
            title: 'Welcome to ' . \App\Helpers\SettingsHelper::bankName(),
            message: "Welcome to your new banking account! We're excited to have you on board. Please complete your profile and KYC verification to get started.",
            data: [
                'welcome_date' => now()->toISOString(),
            ]
        );
    }

    /**
     * KYC approval notification.
     */
    public function notifyKycApproved(User $user): void
    {
        $this->createNotification(
            user: $user,
            type: 'kyc_approved',
            title: 'KYC Verification Approved',
            message: "Congratulations! Your KYC verification has been approved. You now have full access to all banking services.",
            data: [
                'approved_at' => now()->toISOString(),
            ]
        );
    }

    /**
     * KYC rejection notification.
     */
    public function notifyKycRejected(User $user, string $reason = null): void
    {
        $this->createNotification(
            user: $user,
            type: 'kyc_rejected',
            title: 'KYC Verification Rejected',
            message: "Your KYC verification was rejected" . ($reason ? " due to: {$reason}" : '') . ". Please review and resubmit your documents.",
            data: [
                'rejected_at' => now()->toISOString(),
                'reason' => $reason,
            ]
        );
    }

    /**
     * Virtual card creation notification.
     */
    public function notifyVirtualCardCreated(User $user, string $cardType, string $lastFourDigits, array $additionalData = []): void
    {
        $this->createNotification(
            user: $user,
            type: 'virtual_card_created',
            title: 'Virtual Card Created',
            message: "Your new {$cardType} virtual card ending in {$lastFourDigits} has been created successfully and is ready to use.",
            data: array_merge([
                'card_type' => $cardType,
                'last_four_digits' => $lastFourDigits,
                'created_at' => now()->toISOString(),
            ], $additionalData)
        );
    }

    /**
     * Transfer completion notification.
     */
    public function notifyTransferCompleted(User $user, float $amount, string $currency, string $recipient, string $reference): void
    {
        $this->createNotification(
            user: $user,
            type: 'transfer_completed',
            title: 'Transfer Completed',
            message: "Your transfer of " . number_format($amount, 2) . " {$currency} to {$recipient} has been completed successfully. Reference: {$reference}",
            data: [
                'amount' => $amount,
                'currency' => $currency,
                'recipient' => $recipient,
                'reference' => $reference,
                'completed_at' => now()->toISOString(),
            ]
        );
    }

    /**
     * Inter-account transfer notification.
     */
    public function notifyInterAccountTransfer(User $user, float $amount, string $currency, string $fromAccount, string $toAccount): void
    {
        $this->createNotification(
            user: $user,
            type: 'inter_account_transfer',
            title: 'Inter-Account Transfer',
            message: "Transfer of " . number_format($amount, 2) . " {$currency} from {$fromAccount} to {$toAccount} has been completed.",
            data: [
                'amount' => $amount,
                'currency' => $currency,
                'from_account' => $fromAccount,
                'to_account' => $toAccount,
                'transferred_at' => now()->toISOString(),
            ]
        );
    }

    /**
     * Cheque deposit notification.
     */
    public function notifyChequeDeposit(User $user, float $amount, string $currency, string $status): void
    {
        $this->createNotification(
            user: $user,
            type: 'cheque_deposit',
            title: 'Cheque Deposit ' . ucfirst($status),
            message: "Your cheque deposit of " . number_format($amount, 2) . " {$currency} has been {$status}.",
            data: [
                'amount' => $amount,
                'currency' => $currency,
                'status' => $status,
                'processed_at' => now()->toISOString(),
            ]
        );
    }

    /**
     * Password change notification.
     */
    public function notifyPasswordChanged(User $user): void
    {
        // Create in-app notification
        $this->createNotification(
            user: $user,
            type: 'password_changed',
            title: 'Password Changed',
            message: "Your account password has been successfully changed. If you didn't make this change, please contact support immediately.",
            data: [
                'changed_at' => now()->toISOString(),
            ],
            sendEmail: false // We'll send custom email separately
        );

        // Send custom password changed email notification
        try {
            $user->notify(new \App\Notifications\PasswordChangedNotification);
        } catch (\Exception $e) {
            Log::error('Failed to send password changed email notification: ' . $e->getMessage());
        }
    }

    /**
     * PIN change notification.
     */
    public function notifyPinChanged(User $user): void
    {
        $this->createNotification(
            user: $user,
            type: 'pin_changed',
            title: 'PIN Changed',
            message: "Your transaction PIN has been successfully changed. If you didn't make this change, please contact support immediately.",
            data: [
                'changed_at' => now()->toISOString(),
            ]
        );
    }

    /**
     * Beneficiary added notification.
     */
    public function notifyBeneficiaryAdded(User $user, string $beneficiaryName, string $accountNumber): void
    {
        $this->createNotification(
            user: $user,
            type: 'beneficiary_added',
            title: 'New Beneficiary Added',
            message: "A new beneficiary '{$beneficiaryName}' with account number {$accountNumber} has been added to your account.",
            data: [
                'beneficiary_name' => $beneficiaryName,
                'account_number' => $accountNumber,
                'added_at' => now()->toISOString(),
            ]
        );
    }

    /**
     * Two-factor authentication enabled notification.
     */
    public function notifyTwoFactorEnabled(User $user, string $method): void
    {
        $this->createNotification(
            user: $user,
            type: 'two_factor_enabled',
            title: 'Two-Factor Authentication Enabled',
            message: "Two-factor authentication has been enabled for your account using {$method}. Your account is now more secure.",
            data: [
                'method' => $method,
                'enabled_at' => now()->toISOString(),
            ]
        );
    }

    /**
     * Two-factor authentication disabled notification.
     */
    public function notifyTwoFactorDisabled(User $user): void
    {
        $this->createNotification(
            user: $user,
            type: 'two_factor_disabled',
            title: 'Two-Factor Authentication Disabled',
            message: "Two-factor authentication has been disabled for your account. Please consider re-enabling it for better security.",
            data: [
                'disabled_at' => now()->toISOString(),
            ]
        );
    }

    /**
     * Account creation notification.
     */
    public function notifyAccountCreated(User $user, string $accountNumber, string $accountType, string $currency, bool $isMasterAccount = false): void
    {
        $this->createNotification(
            user: $user,
            type: 'account_created',
            title: 'New Account Created',
            message: "Your new {$accountType} account ({$accountNumber}) has been created successfully" . ($isMasterAccount ? " as your master account" : '') . " and is pending verification.",
            data: [
                'account_number' => $accountNumber,
                'account_type' => $accountType,
                'currency' => $currency,
                'is_master_account' => $isMasterAccount,
                'created_at' => now()->toISOString(),
            ]
        );
    }

    /**
     * Get unread notifications count for a user.
     */
    public function getUnreadCount(User $user): int
    {
        return $user->notifications()->unread()->count();
    }

    /**
     * Mark all notifications as read for a user.
     */
    public function markAllAsRead(User $user): void
    {
        $user->notifications()->unread()->update([
            'is_read' => true,
            'read_at' => now(),
        ]);
    }

    /**
     * Get recent notifications for a user.
     */
    public function getRecentNotifications(User $user, int $limit = 10)
    {
        return $user->notifications()
            ->orderBy('created_at', 'desc')
            ->limit($limit)
            ->get();
    }
}
