<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class ExchangeRateService
{
    private string $baseUrl;
    private string $appId;
    private int $cacheDuration = 3600;

    public function __construct()
    {
        $this->baseUrl = config('services.openexchange.base_url', 'https://openexchangerates.org/api');
        $this->appId = config('services.openexchange.app_id');
    }

    /**
     * Get latest exchange rates
     */
    public function getLatestRates(string $baseCurrency = 'USD'): array
    {
        $cacheKey = "exchange_rates_{$baseCurrency}_" . date('Y-m-d');
        
        return Cache::remember($cacheKey, $this->cacheDuration, function () use ($baseCurrency) {
            try {
                $response = Http::get("{$this->baseUrl}/latest.json", [
                    'app_id' => $this->appId,
                    'base' => $baseCurrency,
                ]);

                if ($response->successful()) {
                    $data = $response->json();
                    return [
                        'success' => true,
                        'base' => $data['base'],
                        'rates' => $data['rates'],
                        'timestamp' => $data['timestamp'],
                    ];
                }

                Log::error('OpenExchangeRate API error', [
                    'status' => $response->status(),
                    'response' => $response->body()
                ]);

                return [
                    'success' => false,
                    'error' => 'Failed to fetch exchange rates'
                ];

            } catch (\Exception $e) {
                Log::error('Exchange rate service error', [
                    'message' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);

                return [
                    'success' => false,
                    'error' => 'Service temporarily unavailable'
                ];
            }
        });
    }

    /**
     * Convert amount between currencies
     */
    public function convertCurrency(float $amount, string $fromCurrency, string $toCurrency): array
    {
        if ($fromCurrency === $toCurrency) {
            return [
                'success' => true,
                'original_amount' => $amount,
                'converted_amount' => $amount,
                'exchange_rate' => 1.0,
                'from_currency' => $fromCurrency,
                'to_currency' => $toCurrency,
            ];
        }

        // Always use USD as base currency since free plan doesn't allow changing base
        $rates = $this->getLatestRates('USD');
        
        if (!$rates['success']) {
            return $rates;
        }

        // Check if both currencies are supported
        if (!isset($rates['rates'][$fromCurrency]) || !isset($rates['rates'][$toCurrency])) {
            return [
                'success' => false,
                'error' => "Exchange rate not available for {$fromCurrency} or {$toCurrency}"
            ];
        }

        // Calculate cross-rate conversion
        // OpenExchangeRates returns rates where base is USD
        // rates['EUR'] = 0.858322 means 1 USD = 0.858322 EUR
        
        if ($fromCurrency === 'USD') {
            // Converting FROM USD: use the rate directly
            $exchangeRate = $rates['rates'][$toCurrency];
        } elseif ($toCurrency === 'USD') {
            // Converting TO USD: use the inverse of the rate
            $exchangeRate = 1 / $rates['rates'][$fromCurrency];
        } else {
            // Converting between two non-USD currencies: cross-rate calculation
            // Convert from source currency to USD, then from USD to target currency
            $fromToUsd = 1 / $rates['rates'][$fromCurrency]; // How many USD = 1 source currency
            $usdToTarget = $rates['rates'][$toCurrency]; // How many target currency = 1 USD
            $exchangeRate = $fromToUsd * $usdToTarget;
        }
        
        $convertedAmount = $amount * $exchangeRate;

        return [
            'success' => true,
            'original_amount' => $amount,
            'converted_amount' => $convertedAmount,
            'exchange_rate' => $exchangeRate,
            'from_currency' => $fromCurrency,
            'to_currency' => $toCurrency,
            'timestamp' => $rates['timestamp'],
        ];
    }

    /**
     * Get supported currencies
     */
    public function getSupportedCurrencies(): array
    {
        $cacheKey = 'supported_currencies';
        
        return Cache::remember($cacheKey, $this->cacheDuration * 24, function () {
            try {
                $response = Http::get("{$this->baseUrl}/currencies.json");

                if ($response->successful()) {
                    return [
                        'success' => true,
                        'currencies' => $response->json()
                    ];
                }

                return [
                    'success' => false,
                    'error' => 'Failed to fetch currencies'
                ];

            } catch (\Exception $e) {
                Log::error('Currency fetch error', [
                    'message' => $e->getMessage()
                ]);

                return [
                    'success' => false,
                    'error' => 'Service temporarily unavailable'
                ];
            }
        });
    }

    /**
     * Get exchange rate between two currencies
     */
    public function getExchangeRate(string $fromCurrency, string $toCurrency): array
    {
        if ($fromCurrency === $toCurrency) {
            return [
                'success' => true,
                'rate' => 1.0,
                'from_currency' => $fromCurrency,
                'to_currency' => $toCurrency,
            ];
        }

        // Always use USD as base currency since free plan doesn't allow changing base
        $rates = $this->getLatestRates('USD');
        
        if (!$rates['success']) {
            return $rates;
        }

        // Check if both currencies are supported
        if (!isset($rates['rates'][$fromCurrency]) || !isset($rates['rates'][$toCurrency])) {
            return [
                'success' => false,
                'error' => "Exchange rate not available for {$fromCurrency} or {$toCurrency}"
            ];
        }

        // Calculate cross-rate
        // OpenExchangeRates returns rates where base is USD
        // rates['EUR'] = 0.858322 means 1 USD = 0.858322 EUR
        
        if ($fromCurrency === 'USD') {
            // Converting FROM USD: use the rate directly
            $exchangeRate = $rates['rates'][$toCurrency];
        } elseif ($toCurrency === 'USD') {
            // Converting TO USD: use the inverse of the rate
            $exchangeRate = 1 / $rates['rates'][$fromCurrency];
        } else {
            // Converting between two non-USD currencies: cross-rate calculation
            // Convert from source currency to USD, then from USD to target currency
            $fromToUsd = 1 / $rates['rates'][$fromCurrency]; // How many USD = 1 source currency
            $usdToTarget = $rates['rates'][$toCurrency]; // How many target currency = 1 USD
            $exchangeRate = $fromToUsd * $usdToTarget;
        }

        return [
            'success' => true,
            'rate' => $exchangeRate,
            'from_currency' => $fromCurrency,
            'to_currency' => $toCurrency,
            'timestamp' => $rates['timestamp'],
        ];
    }

    /**
     * Clear cache for testing
     */
    public function clearCache(): void
    {
        Cache::forget('exchange_rates_USD_' . date('Y-m-d'));
        Cache::forget('supported_currencies');
    }
}
