<?php

namespace App\Services\Admin;

use App\Models\Setting;
use App\Models\Country;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\Request;

class AdminSettingService
{
    /**
     * Get all system settings
     */
    public function getAllSettings()
    {
        return Cache::remember('admin_settings', 3600, function () {
            return [
                'bank' => $this->getBankSettings(),
                'limits' => $this->getTransactionLimits(),
                'compliance' => $this->getComplianceSettings(),
                'notifications' => $this->getNotificationSettings(),
                'system' => $this->getSystemSettings(),
            ];
        });
    }

    /**
     * Get bank settings
     */
    public function getBankSettings()
    {
        return Setting::getGroup('bank');
    }

    /**
     * Get transaction limits settings
     */
    public function getTransactionLimits()
    {
        return Setting::getGroup('limits');
    }

    /**
     * Get compliance settings
     */
    public function getComplianceSettings()
    {
        return Setting::getGroup('compliance');
    }

    /**
     * Get notification settings
     */
    public function getNotificationSettings()
    {
        return Setting::getGroup('notifications');
    }

    /**
     * Get system settings
     */
    public function getSystemSettings()
    {
        return Setting::getGroup('system');
    }

    /**
     * Update bank settings
     */
 public function updateBankSettings(Request $request)
    {
        $settings = $request->except(['_token', 'logo', 'favicon']);

        // Handle logo upload
        if ($request->hasFile('logo')) {
            $logoPath = $this->handleFileUpload($request->file('logo'), 'logos', 'bank-logo');
            if ($logoPath) {
                $settings['logo_url'] = asset($logoPath);
            }
        }

        // Handle favicon upload
        if ($request->hasFile('favicon')) {
            $faviconPath = $this->handleFileUpload($request->file('favicon'), 'favicons', 'bank-favicon');
            if ($faviconPath) {
                $settings['favicon_url'] = asset($faviconPath);
            }
        }

        foreach ($settings as $key => $value) {
            Setting::set($key, $value, 'string', 'bank');
        }

        Cache::forget('admin_settings');
        Setting::clearCache();

        return true;
    }

    private function handleFileUpload($file, $folder, $prefix)
    {
        if (!$file || !$file->isValid()) {
            return null;
        }

        $extension = $file->getClientOriginalExtension();
        $filename = $prefix . '-' . time() . '.' . $extension;

        $destinationPath = public_path($folder);
        if (!file_exists($destinationPath)) {
            mkdir($destinationPath, 0755, true);
        }

        $file->move($destinationPath, $filename);

        return $folder . '/' . $filename;
    }


    /**
     * Update transaction limits
     */
    public function updateTransactionLimits(array $settings)
    {
        foreach ($settings as $key => $value) {
            $type = in_array($key, ['max_daily_transactions', 'max_monthly_transactions']) ? 'integer' : 'float';
            Setting::set($key, $value, $type, 'limits');
        }

        Cache::forget('admin_settings');
        Setting::clearCache();
        return true;
    }

    /**
     * Update compliance settings
     */
    public function updateComplianceSettings(array $settings)
    {
        foreach ($settings as $key => $value) {
            $type = in_array($key, ['suspicious_activity_threshold', 'kyc_auto_approve_threshold']) ? 'float' : 
                   (in_array($key, ['compliance_retention_days']) ? 'integer' : 
                   (in_array($key, ['high_risk_countries', 'sanctioned_countries']) ? 'array' : 'boolean'));
            Setting::set($key, $value, $type, 'compliance');
        }

        Cache::forget('admin_settings');
        Setting::clearCache();
        return true;
    }

    /**
     * Update notification settings
     */
    public function updateNotificationSettings(array $settings)
    {
        foreach ($settings as $key => $value) {
            $type = in_array($key, ['smtp_port', 'email_verification_enabled']) ? 'integer' : 'string';
            Setting::set($key, $value, $type, 'notifications');
        }

        Cache::forget('admin_settings');
        Setting::clearCache();
        return true;
    }

    /**
     * Test SMTP configuration
     */
    public function testSmtpConfiguration(array $config)
    {
        try {
            // Create a temporary mail configuration
            $mailConfig = [
                'driver' => $config['smtp_driver'],
                'host' => $config['smtp_host'],
                'port' => $config['smtp_port'],
                'username' => $config['smtp_username'],
                'password' => $config['smtp_password'],
                'encryption' => $config['smtp_encryption'] ?? null,
                'from' => [
                    'address' => $config['from_email'],
                    'name' => $config['from_name'],
                ],
            ];

            // Set the mail configuration temporarily
            config(['mail' => array_merge(config('mail'), $mailConfig)]);

            // Test sending an email
            \Illuminate\Support\Facades\Mail::raw('This is a test email to verify SMTP configuration.', function ($message) use ($config) {
                $message->to($config['from_email'])
                        ->subject('SMTP Configuration Test')
                        ->from($config['from_email'], $config['from_name']);
            });

            return [
                'success' => true,
                'message' => 'SMTP configuration test successful! Email sent to ' . $config['from_email']
            ];

        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'SMTP test failed: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Update system settings
     */
    public function updateSystemSettings(array $settings)
    {
        foreach ($settings as $key => $value) {
            $type = in_array($key, ['maintenance_mode', 'debug_mode']) ? 'boolean' : 'string';
            Setting::set($key, $value, $type, 'system');
        }

        // Update environment file for debug mode
        if (isset($settings['debug_mode'])) {
            $this->updateEnvironmentFile('APP_DEBUG', $settings['debug_mode'] ? 'true' : 'false');
        }

        // Handle maintenance mode
        if (isset($settings['maintenance_mode'])) {
            if ($settings['maintenance_mode']) {
                $this->enableMaintenanceMode();
            } else {
                $this->disableMaintenanceMode();
            }
        }

        Cache::forget('admin_settings');
        Setting::clearCache();
        return true;
    }

    /**
     * Update environment file
     */
    private function updateEnvironmentFile($key, $value)
    {
        $envFile = base_path('.env');
        
        if (!file_exists($envFile)) {
            return false;
        }

        $envContent = file_get_contents($envFile);
        
        // Check if key exists
        if (preg_match("/^{$key}=.*/m", $envContent)) {
            // Update existing key
            $envContent = preg_replace("/^{$key}=.*/m", "{$key}={$value}", $envContent);
        } else {
            // Add new key
            $envContent .= "\n{$key}={$value}";
        }
        
        return file_put_contents($envFile, $envContent) !== false;
    }

    /**
     * Enable maintenance mode
     */
    private function enableMaintenanceMode()
    {
        try {
            \Artisan::call('down', [
                '--retry' => 60,
                '--refresh' => 15
            ]);
            return true;
        } catch (\Exception $e) {
            \Log::error('Failed to enable maintenance mode: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Disable maintenance mode
     */
    private function disableMaintenanceMode()
    {
        try {
            \Artisan::call('up');
            return true;
        } catch (\Exception $e) {
            \Log::error('Failed to disable maintenance mode: ' . $e->getMessage());
            return false;
        }
    }



    /**
     * Get available timezones
     */
    public function getAvailableTimezones()
    {
        return timezone_identifiers_list();
    }

    /**
     * Get available countries from database
     */
    public function getAvailableCountries()
    {
        return Country::select('iso', 'name')
            ->orderBy('name')
            ->get()
            ->pluck('name', 'iso')
            ->toArray();
    }

    /**
     * Get countries with currencies
     */
    public function getCountriesWithCurrencies()
    {
        return Country::withCurrency()
            ->select('iso', 'name', 'currency', 'currency_name', 'currency_symbol')
            ->orderBy('name')
            ->get();
    }

    /**
     * Get high risk countries for compliance settings
     */
    public function getHighRiskCountries()
    {
        return Country::select('iso', 'name')
            ->whereIn('iso', ['AF', 'IR', 'KP', 'SY', 'VE', 'ZW']) // Example high-risk countries
            ->orderBy('name')
            ->get()
            ->pluck('name', 'iso')
            ->toArray();
    }

    /**
     * Get sanctioned countries for compliance settings
     */
    public function getSanctionedCountries()
    {
        return Country::select('iso', 'name')
            ->whereIn('iso', ['IR', 'KP', 'SY', 'VE']) // Example sanctioned countries
            ->orderBy('name')
            ->get()
            ->pluck('name', 'iso')
            ->toArray();
    }

    /**
     * Test email configuration
     */
    public function testEmailConfiguration(array $settings)
    {
        try {
            // This would test the email configuration
            // For now, we'll just return true
            return ['success' => true, 'message' => 'Email configuration test successful'];
        } catch (\Exception $e) {
            return ['success' => false, 'message' => 'Email configuration test failed: ' . $e->getMessage()];
        }
    }

    /**
     * Test SMS configuration
     */
    public function testSmsConfiguration(array $settings)
    {
        try {
            // This would test the SMS configuration
            // For now, we'll just return true
            return ['success' => true, 'message' => 'SMS configuration test successful'];
        } catch (\Exception $e) {
            return ['success' => false, 'message' => 'SMS configuration test failed: ' . $e->getMessage()];
        }
    }
}
