<?php

namespace App\Services\Admin;

use App\Models\User;
use App\Models\Account;
use App\Models\Transaction;
use App\Models\KycDocument;
use App\Models\ComplianceLog;
use App\Models\UserProfile;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class AdminReportService
{
    /**
     * Get user registration report
     */
    public function getUserRegistrationReport($startDate = null, $endDate = null, $groupBy = 'day')
    {
        $query = User::query();

        if ($startDate) {
            $query->whereDate('created_at', '>=', $startDate);
        }

        if ($endDate) {
            $query->whereDate('created_at', '<=', $endDate);
        }

        $format = $this->getDateFormat($groupBy);
        
        return $query->selectRaw("DATE_FORMAT(created_at, '{$format}') as period, COUNT(*) as count")
            ->groupBy('period')
            ->orderBy('period')
            ->get();
    }

    /**
     * Get user activity report
     */
    public function getUserActivityReport($startDate = null, $endDate = null)
    {
        $query = User::withCount(['accounts', 'kycDocuments', 'transactions']);

        if ($startDate) {
            $query->whereDate('created_at', '>=', $startDate);
        }

        if ($endDate) {
            $query->whereDate('created_at', '<=', $endDate);
        }

        return $query->get();
    }

    /**
     * Get user status report
     */
    public function getUserStatusReport()
    {
        return [
            'total_users' => User::count(),
            'active_users' => User::where('customer_status', 'active')->count(),
            'inactive_users' => User::where('customer_status', 'inactive')->count(),
            'verified_users' => User::whereHas('profile', function($q) {
                $q->where('kyc_status', 'verified');
            })->count(),
            'pending_kyc' => User::whereHas('profile', function($q) {
                $q->whereIn('kyc_status', ['pending', 'under_review']);
            })->count(),
            'rejected_kyc' => User::whereHas('profile', function($q) {
                $q->where('kyc_status', 'rejected');
            })->count(),
        ];
    }

    /**
     * Get account creation report
     */
    public function getAccountCreationReport($startDate = null, $endDate = null, $groupBy = 'day')
    {
        $query = Account::query();

        if ($startDate) {
            $query->whereDate('created_at', '>=', $startDate);
        }

        if ($endDate) {
            $query->whereDate('created_at', '<=', $endDate);
        }

        $format = $this->getDateFormat($groupBy);
        
        return $query->selectRaw("DATE_FORMAT(created_at, '{$format}') as period, COUNT(*) as count")
            ->groupBy('period')
            ->orderBy('period')
            ->get();
    }

    /**
     * Get account status report
     */
    public function getAccountStatusReport()
    {
        return [
            'total_accounts' => Account::count(),
            'active_accounts' => Account::where('status', 'active')->count(),
            'suspended_accounts' => Account::where('status', 'suspended')->count(),
            'closed_accounts' => Account::where('status', 'closed')->count(),
            'savings_accounts' => Account::where('account_type', 'savings')->count(),
            'checking_accounts' => Account::where('account_type', 'checking')->count(),
            'business_accounts' => Account::where('account_type', 'business')->count(),
        ];
    }

    /**
     * Get transaction volume report
     */
    public function getTransactionVolumeReport($startDate = null, $endDate = null, $groupBy = 'day')
    {
        $query = Transaction::query();

        if ($startDate) {
            $query->whereDate('created_at', '>=', $startDate);
        }

        if ($endDate) {
            $query->whereDate('created_at', '<=', $endDate);
        }

        $format = $this->getDateFormat($groupBy);
        
        return $query->selectRaw("
                DATE_FORMAT(created_at, '{$format}') as period, 
                COUNT(*) as count,
                SUM(amount) as total_amount,
                AVG(amount) as avg_amount
            ")
            ->groupBy('period')
            ->orderBy('period')
            ->get();
    }

    /**
     * Get transaction type report
     */
    public function getTransactionTypeReport($startDate = null, $endDate = null)
    {
        $query = Transaction::selectRaw('type, COUNT(*) as count, SUM(amount) as total_amount')
            ->groupBy('type');

        if ($startDate) {
            $query->whereDate('created_at', '>=', $startDate);
        }

        if ($endDate) {
            $query->whereDate('created_at', '<=', $endDate);
        }

        return $query->orderBy('count', 'desc')->get();
    }

    /**
     * Get transaction status report
     */
    public function getTransactionStatusReport($startDate = null, $endDate = null)
    {
        $query = Transaction::selectRaw('status, COUNT(*) as count, SUM(amount) as total_amount')
            ->groupBy('status');

        if ($startDate) {
            $query->whereDate('created_at', '>=', $startDate);
        }

        if ($endDate) {
            $query->whereDate('created_at', '<=', $endDate);
        }

        return $query->orderBy('count', 'desc')->get();
    }

    /**
     * Get KYC submission report
     */
    public function getKycSubmissionReport($startDate = null, $endDate = null, $groupBy = 'day')
    {
        $query = KycDocument::query();

        if ($startDate) {
            $query->whereDate('created_at', '>=', $startDate);
        }

        if ($endDate) {
            $query->whereDate('created_at', '<=', $endDate);
        }

        $format = $this->getDateFormat($groupBy);
        
        return $query->selectRaw("DATE_FORMAT(created_at, '{$format}') as period, COUNT(*) as count")
            ->groupBy('period')
            ->orderBy('period')
            ->get();
    }

    /**
     * Get KYC status report
     */
    public function getKycStatusReport()
    {
        return [
            'total_documents' => KycDocument::count(),
            'pending_documents' => KycDocument::where('status', 'pending')->count(),
            'approved_documents' => KycDocument::where('status', 'approved')->count(),
            'rejected_documents' => KycDocument::where('status', 'rejected')->count(),
            'expired_documents' => KycDocument::where('status', 'expired')->count(),
            'passport_documents' => KycDocument::where('document_type', 'passport')->count(),
            'drivers_license' => KycDocument::where('document_type', 'drivers_license')->count(),
            'national_id' => KycDocument::where('document_type', 'national_id')->count(),
            'utility_bill' => KycDocument::where('document_type', 'utility_bill')->count(),
            'bank_statement' => KycDocument::where('document_type', 'bank_statement')->count(),
        ];
    }

    /**
     * Get compliance flag report
     */
    public function getComplianceFlagReport($startDate = null, $endDate = null, $groupBy = 'day')
    {
        $query = ComplianceLog::query();

        if ($startDate) {
            $query->whereDate('created_at', '>=', $startDate);
        }

        if ($endDate) {
            $query->whereDate('created_at', '<=', $endDate);
        }

        $format = $this->getDateFormat($groupBy);
        
        return $query->selectRaw("DATE_FORMAT(created_at, '{$format}') as period, COUNT(*) as count")
            ->groupBy('period')
            ->orderBy('period')
            ->get();
    }

    /**
     * Get compliance status report
     */
    public function getComplianceStatusReport()
    {
        return [
            'total_logs' => ComplianceLog::count(),
            'pending_logs' => ComplianceLog::where('status', 'pending')->count(),
            'under_review' => ComplianceLog::where('status', 'under_review')->count(),
            'approved_logs' => ComplianceLog::where('status', 'approved')->count(),
            'rejected_logs' => ComplianceLog::where('status', 'rejected')->count(),
            'escalated_logs' => ComplianceLog::where('status', 'escalated')->count(),
            'critical_severity' => ComplianceLog::where('severity', 'critical')->count(),
            'high_severity' => ComplianceLog::where('severity', 'high')->count(),
            'medium_severity' => ComplianceLog::where('severity', 'medium')->count(),
            'low_severity' => ComplianceLog::where('severity', 'low')->count(),
        ];
    }

    /**
     * Get compliance type report
     */
    public function getComplianceTypeReport($startDate = null, $endDate = null)
    {
        $query = ComplianceLog::selectRaw('type, COUNT(*) as count')
            ->groupBy('type');

        if ($startDate) {
            $query->whereDate('created_at', '>=', $startDate);
        }

        if ($endDate) {
            $query->whereDate('created_at', '<=', $endDate);
        }

        return $query->orderBy('count', 'desc')->get();
    }

    /**
     * Get dashboard summary
     */
    public function getDashboardSummary()
    {
        return [
            'users' => [
                'total' => User::count(),
                'active' => User::where('customer_status', 'active')->count(),
                'new_today' => User::whereDate('created_at', today())->count(),
                'new_this_month' => User::whereMonth('created_at', now()->month)->count(),
            ],
            'accounts' => [
                'total' => Account::count(),
                'active' => Account::where('status', 'active')->count(),
                'new_today' => Account::whereDate('created_at', today())->count(),
                'new_this_month' => Account::whereMonth('created_at', now()->month)->count(),
            ],
            'transactions' => [
                'total' => Transaction::count(),
                'today' => Transaction::whereDate('created_at', today())->count(),
                'this_month' => Transaction::whereMonth('created_at', now()->month)->count(),
                'total_volume' => Transaction::sum('amount'),
                'today_volume' => Transaction::whereDate('created_at', today())->sum('amount'),
            ],
            'kyc' => [
                'total_documents' => KycDocument::count(),
                'pending' => KycDocument::where('status', 'pending')->count(),
                'approved' => KycDocument::where('status', 'approved')->count(),
                'rejected' => KycDocument::where('status', 'rejected')->count(),
            ],
            'compliance' => [
                'total_logs' => ComplianceLog::count(),
                'pending' => ComplianceLog::where('status', 'pending')->count(),
                'high_priority' => ComplianceLog::whereIn('severity', ['high', 'critical'])
                    ->whereIn('status', ['pending', 'under_review'])
                    ->count(),
            ],
        ];
    }

    /**
     * Get date format based on group by parameter
     */
    private function getDateFormat($groupBy)
    {
        return match($groupBy) {
            'hour' => '%Y-%m-%d %H:00:00',
            'day' => '%Y-%m-%d',
            'week' => '%Y-%u',
            'month' => '%Y-%m',
            'year' => '%Y',
            default => '%Y-%m-%d',
        };
    }

    /**
     * Export data to CSV
     */
    public function exportToCsv($data, $filename)
    {
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];

        $callback = function() use ($data) {
            $file = fopen('php://output', 'w');
            
            if (!empty($data)) {
                // Write headers
                fputcsv($file, array_keys($data[0]));
                
                // Write data
                foreach ($data as $row) {
                    fputcsv($file, $row);
                }
            }
            
            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}
