<?php

namespace App\Services\Admin;

use App\Models\User;
use App\Models\Account;
use App\Models\Transaction;
use App\Models\KycDocument;
use App\Models\ComplianceLog;
use App\Models\UserProfile;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class AdminAnalyticsService
{
    /**
     * Get user growth analytics
     */
    public function getUserGrowthAnalytics($period = '30d')
    {
        $endDate = now();
        $startDate = $this->getStartDate($period, $endDate);

        $query = User::selectRaw("DATE_FORMAT(created_at, '%Y-%m-%d') as date, COUNT(*) as count")
            ->whereBetween('created_at', [$startDate, $endDate])
            ->groupBy('date')
            ->orderBy('date');

        return $query->get();
    }

    /**
     * Get user activity analytics
     */
    public function getUserActivityAnalytics($period = '30d')
    {
        $endDate = now();
        $startDate = $this->getStartDate($period, $endDate);

        return [
            'total_users' => User::count(),
            'active_users' => User::where('customer_status', 'active')->count(),
            'new_users' => User::whereBetween('created_at', [$startDate, $endDate])->count(),
            'user_growth_rate' => $this->calculateGrowthRate(
                User::where('created_at', '<', $startDate)->count(),
                User::whereBetween('created_at', [$startDate, $endDate])->count()
            ),
            'login_activity' => $this->getLoginActivity($startDate, $endDate),
            'user_retention' => $this->getUserRetention($startDate, $endDate),
        ];
    }

    /**
     * Get user demographics analytics
     */
    public function getUserDemographicsAnalytics()
    {
        return [
            'by_customer_type' => User::selectRaw('customer_type, COUNT(*) as count')
                ->groupBy('customer_type')
                ->get()
                ->pluck('count', 'customer_type'),
            
            'by_customer_status' => User::selectRaw('customer_status, COUNT(*) as count')
                ->groupBy('customer_status')
                ->get()
                ->pluck('count', 'customer_status'),
            
            'by_kyc_status' => User::join('user_profiles', 'users.id', '=', 'user_profiles.user_id')
                ->selectRaw('user_profiles.kyc_status, COUNT(*) as count')
                ->groupBy('user_profiles.kyc_status')
                ->get()
                ->pluck('count', 'kyc_status'),
            
            'by_country' => User::join('user_profiles', 'users.id', '=', 'user_profiles.user_id')
                ->selectRaw('user_profiles.country, COUNT(*) as count')
                ->groupBy('user_profiles.country')
                ->orderBy('count', 'desc')
                ->limit(10)
                ->get()
                ->pluck('count', 'country'),
            
            'by_age_group' => $this->getAgeGroupDistribution(),
        ];
    }

    /**
     * Get transaction volume analytics
     */
    public function getTransactionVolumeAnalytics($period = '30d')
    {
        $endDate = now();
        $startDate = $this->getStartDate($period, $endDate);

        $query = Transaction::selectRaw("DATE_FORMAT(created_at, '%Y-%m-%d') as date, COUNT(*) as count, SUM(amount) as total_amount")
            ->whereBetween('created_at', [$startDate, $endDate])
            ->groupBy('date')
            ->orderBy('date');

        return $query->get();
    }

    /**
     * Get transaction trends analytics
     */
    public function getTransactionTrendsAnalytics($period = '30d')
    {
        $endDate = now();
        $startDate = $this->getStartDate($period, $endDate);

        return [
            'total_transactions' => Transaction::count(),
            'period_transactions' => Transaction::whereBetween('created_at', [$startDate, $endDate])->count(),
            'total_volume' => Transaction::sum('amount'),
            'period_volume' => Transaction::whereBetween('created_at', [$startDate, $endDate])->sum('amount'),
            'average_transaction' => Transaction::whereBetween('created_at', [$startDate, $endDate])->avg('amount'),
            'transaction_growth_rate' => $this->calculateGrowthRate(
                Transaction::where('created_at', '<', $startDate)->count(),
                Transaction::whereBetween('created_at', [$startDate, $endDate])->count()
            ),
            'volume_growth_rate' => $this->calculateGrowthRate(
                Transaction::where('created_at', '<', $startDate)->sum('amount'),
                Transaction::whereBetween('created_at', [$startDate, $endDate])->sum('amount')
            ),
            'by_type' => Transaction::selectRaw('type, COUNT(*) as count, SUM(amount) as total_amount')
                ->whereBetween('created_at', [$startDate, $endDate])
                ->groupBy('type')
                ->get(),
            'by_status' => Transaction::selectRaw('status, COUNT(*) as count, SUM(amount) as total_amount')
                ->whereBetween('created_at', [$startDate, $endDate])
                ->groupBy('status')
                ->get(),
        ];
    }

    /**
     * Get account creation analytics
     */
    public function getAccountCreationAnalytics($period = '30d')
    {
        $endDate = now();
        $startDate = $this->getStartDate($period, $endDate);

        $query = Account::selectRaw("DATE_FORMAT(created_at, '%Y-%m-%d') as date, COUNT(*) as count")
            ->whereBetween('created_at', [$startDate, $endDate])
            ->groupBy('date')
            ->orderBy('date');

        return $query->get();
    }

    /**
     * Get account performance analytics
     */
    public function getAccountPerformanceAnalytics()
    {
        return [
            'total_accounts' => Account::count(),
            'active_accounts' => Account::where('status', 'active')->count(),
            'suspended_accounts' => Account::where('status', 'suspended')->count(),
            'closed_accounts' => Account::where('status', 'closed')->count(),
            'by_type' => Account::selectRaw('account_type, COUNT(*) as count')
                ->groupBy('account_type')
                ->get()
                ->pluck('count', 'account_type'),
            'by_status' => Account::selectRaw('status, COUNT(*) as count')
                ->groupBy('status')
                ->get()
                ->pluck('count', 'status'),
            'average_balance' => Account::where('status', 'active')->avg('balance'),
            'total_balance' => Account::where('status', 'active')->sum('balance'),
        ];
    }

    /**
     * Get KYC submission analytics
     */
    public function getKycSubmissionAnalytics($period = '30d')
    {
        $endDate = now();
        $startDate = $this->getStartDate($period, $endDate);

        $query = KycDocument::selectRaw("DATE_FORMAT(created_at, '%Y-%m-%d') as date, COUNT(*) as count")
            ->whereBetween('created_at', [$startDate, $endDate])
            ->groupBy('date')
            ->orderBy('date');

        return $query->get();
    }

    /**
     * Get KYC processing analytics
     */
    public function getKycProcessingAnalytics()
    {
        return [
            'total_documents' => KycDocument::count(),
            'pending_documents' => KycDocument::where('status', 'pending')->count(),
            'approved_documents' => KycDocument::where('status', 'approved')->count(),
            'rejected_documents' => KycDocument::where('status', 'rejected')->count(),
            'expired_documents' => KycDocument::where('status', 'expired')->count(),
            'by_type' => KycDocument::selectRaw('document_type, COUNT(*) as count')
                ->groupBy('document_type')
                ->get()
                ->pluck('count', 'document_type'),
            'by_status' => KycDocument::selectRaw('status, COUNT(*) as count')
                ->groupBy('status')
                ->get()
                ->pluck('count', 'status'),
            'average_processing_time' => $this->getAverageProcessingTime(),
            'approval_rate' => $this->getKycApprovalRate(),
        ];
    }

    /**
     * Get compliance flag analytics
     */
    public function getComplianceFlagAnalytics($period = '30d')
    {
        $endDate = now();
        $startDate = $this->getStartDate($period, $endDate);

        $query = ComplianceLog::selectRaw("DATE_FORMAT(created_at, '%Y-%m-%d') as date, COUNT(*) as count")
            ->whereBetween('created_at', [$startDate, $endDate])
            ->groupBy('date')
            ->orderBy('date');

        return $query->get();
    }

    /**
     * Get compliance monitoring analytics
     */
    public function getComplianceMonitoringAnalytics()
    {
        return [
            'total_logs' => ComplianceLog::count(),
            'pending_logs' => ComplianceLog::where('status', 'pending')->count(),
            'under_review' => ComplianceLog::where('status', 'under_review')->count(),
            'approved_logs' => ComplianceLog::where('status', 'approved')->count(),
            'rejected_logs' => ComplianceLog::where('status', 'rejected')->count(),
            'escalated_logs' => ComplianceLog::where('status', 'escalated')->count(),
            'by_severity' => ComplianceLog::selectRaw('severity, COUNT(*) as count')
                ->groupBy('severity')
                ->get()
                ->pluck('count', 'severity'),
            'by_type' => ComplianceLog::selectRaw('type, COUNT(*) as count')
                ->groupBy('type')
                ->get()
                ->pluck('count', 'type'),
            'by_status' => ComplianceLog::selectRaw('status, COUNT(*) as count')
                ->groupBy('status')
                ->get()
                ->pluck('count', 'status'),
            'critical_issues' => ComplianceLog::where('severity', 'critical')->count(),
            'high_priority' => ComplianceLog::where('severity', 'high')->count(),
        ];
    }

    /**
     * Get real-time dashboard metrics
     */
    public function getRealTimeMetrics()
    {
        $today = now()->startOfDay();
        $yesterday = $today->copy()->subDay();
        $thisWeek = now()->startOfWeek();
        $lastWeek = $thisWeek->copy()->subWeek();
        $thisMonth = now()->startOfMonth();
        $lastMonth = $thisMonth->copy()->subMonth();

        return [
            'users' => [
                'today' => User::whereDate('created_at', $today)->count(),
                'yesterday' => User::whereDate('created_at', $yesterday)->count(),
                'this_week' => User::whereBetween('created_at', [$thisWeek, now()])->count(),
                'last_week' => User::whereBetween('created_at', [$lastWeek, $thisWeek])->count(),
                'this_month' => User::whereBetween('created_at', [$thisMonth, now()])->count(),
                'last_month' => User::whereBetween('created_at', [$lastMonth, $thisMonth])->count(),
            ],
            'transactions' => [
                'today' => Transaction::whereDate('created_at', $today)->count(),
                'yesterday' => Transaction::whereDate('created_at', $yesterday)->count(),
                'this_week' => Transaction::whereBetween('created_at', [$thisWeek, now()])->count(),
                'last_week' => Transaction::whereBetween('created_at', [$lastWeek, $thisWeek])->count(),
                'this_month' => Transaction::whereBetween('created_at', [$thisMonth, now()])->count(),
                'last_month' => Transaction::whereBetween('created_at', [$lastMonth, $thisMonth])->count(),
            ],
            'accounts' => [
                'today' => Account::whereDate('created_at', $today)->count(),
                'yesterday' => Account::whereDate('created_at', $yesterday)->count(),
                'this_week' => Account::whereBetween('created_at', [$thisWeek, now()])->count(),
                'last_week' => Account::whereBetween('created_at', [$lastWeek, $thisWeek])->count(),
                'this_month' => Account::whereBetween('created_at', [$thisMonth, now()])->count(),
                'last_month' => Account::whereBetween('created_at', [$lastMonth, $thisMonth])->count(),
            ],
            'kyc' => [
                'today' => KycDocument::whereDate('created_at', $today)->count(),
                'yesterday' => KycDocument::whereDate('created_at', $yesterday)->count(),
                'this_week' => KycDocument::whereBetween('created_at', [$thisWeek, now()])->count(),
                'last_week' => KycDocument::whereBetween('created_at', [$lastWeek, $thisWeek])->count(),
                'this_month' => KycDocument::whereBetween('created_at', [$thisMonth, now()])->count(),
                'last_month' => KycDocument::whereBetween('created_at', [$lastMonth, $thisMonth])->count(),
            ],
            'compliance' => [
                'today' => ComplianceLog::whereDate('created_at', $today)->count(),
                'yesterday' => ComplianceLog::whereDate('created_at', $yesterday)->count(),
                'this_week' => ComplianceLog::whereBetween('created_at', [$thisWeek, now()])->count(),
                'last_week' => ComplianceLog::whereBetween('created_at', [$lastWeek, $thisWeek])->count(),
                'this_month' => ComplianceLog::whereBetween('created_at', [$thisMonth, now()])->count(),
                'last_month' => ComplianceLog::whereBetween('created_at', [$lastMonth, $thisMonth])->count(),
            ],
        ];
    }

    /**
     * Get start date based on period
     */
    private function getStartDate($period, $endDate)
    {
        return match($period) {
            '7d' => $endDate->copy()->subDays(7),
            '30d' => $endDate->copy()->subDays(30),
            '90d' => $endDate->copy()->subDays(90),
            '1y' => $endDate->copy()->subYear(),
            default => $endDate->copy()->subDays(30),
        };
    }

    /**
     * Calculate growth rate percentage
     */
    private function calculateGrowthRate($previous, $current)
    {
        if ($previous == 0) {
            return $current > 0 ? 100 : 0;
        }
        
        return round((($current - $previous) / $previous) * 100, 2);
    }

    /**
     * Get login activity
     */
    private function getLoginActivity($startDate, $endDate)
    {
        return User::whereNotNull('last_login_at')
            ->whereBetween('last_login_at', [$startDate, $endDate])
            ->count();
    }

    /**
     * Get user retention
     */
    private function getUserRetention($startDate, $endDate)
    {
        $totalUsers = User::where('created_at', '<', $startDate)->count();
        $activeUsers = User::where('created_at', '<', $startDate)
            ->whereNotNull('last_login_at')
            ->where('last_login_at', '>=', $startDate)
            ->count();

        return $totalUsers > 0 ? round(($activeUsers / $totalUsers) * 100, 2) : 0;
    }

    /**
     * Get age group distribution
     */
    private function getAgeGroupDistribution()
    {
        return UserProfile::selectRaw('
            CASE 
                WHEN TIMESTAMPDIFF(YEAR, date_of_birth, CURDATE()) < 18 THEN "Under 18"
                WHEN TIMESTAMPDIFF(YEAR, date_of_birth, CURDATE()) BETWEEN 18 AND 24 THEN "18-24"
                WHEN TIMESTAMPDIFF(YEAR, date_of_birth, CURDATE()) BETWEEN 25 AND 34 THEN "25-34"
                WHEN TIMESTAMPDIFF(YEAR, date_of_birth, CURDATE()) BETWEEN 35 AND 44 THEN "35-44"
                WHEN TIMESTAMPDIFF(YEAR, date_of_birth, CURDATE()) BETWEEN 45 AND 54 THEN "45-54"
                WHEN TIMESTAMPDIFF(YEAR, date_of_birth, CURDATE()) BETWEEN 55 AND 64 THEN "55-64"
                ELSE "65+"
            END as age_group,
            COUNT(*) as count
        ')
        ->groupBy('age_group')
        ->get()
        ->pluck('count', 'age_group');
    }

    /**
     * Get average KYC processing time
     */
    private function getAverageProcessingTime()
    {
        $processed = KycDocument::whereNotNull('reviewed_at')
            ->whereNotNull('created_at')
            ->get();

        if ($processed->isEmpty()) {
            return 0;
        }

        $totalHours = $processed->sum(function ($doc) {
            return $doc->created_at->diffInHours($doc->reviewed_at);
        });

        return round($totalHours / $processed->count(), 2);
    }

    /**
     * Get KYC approval rate
     */
    private function getKycApprovalRate()
    {
        $total = KycDocument::count();
        $approved = KycDocument::where('status', 'approved')->count();

        return $total > 0 ? round(($approved / $total) * 100, 2) : 0;
    }
}
