<?php

namespace App\Providers;

use App\Models\User;
use App\Models\Code;
use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Str;
use Exception;

class CodeServiceProvider extends ServiceProvider
{
    /**
     * Register services.
     */
    public function register(): void
    {
        //
    }

    /**
     * Bootstrap services.
     */
    public function boot(): void
    {
        // Create codes for active users with profile and accounts
        $this->createCodesForActiveUsers();
    }

    /**
     * Create codes for all active users with profile and accounts.
     */
    private function createCodesForActiveUsers(): void
    {
        try {
            // Find all active users who have profile and accounts but no codes
            $users = User::where('customer_status', 'active')
                ->whereHas('profile')
                ->whereHas('accounts')
                ->whereDoesntHave('code')
                ->get();
            
            foreach ($users as $user) {
                $this->createCodesForUser($user);
            }
            
        } catch (Exception $e) {
            // Silently handle errors to prevent application crashes
        }
    }

    /**
     * Create codes record for user with profile and account.
     */
    private function createCodesForUser(User $user): void
    {
        try {
            // Generate unique codes
            $cotCode = $this->generateUniqueCode('COT');
            $taxCode = $this->generateUniqueCode('TAX');
            $imfCode = $this->generateUniqueCode('IMF');

            // Create codes record
            Code::create([
                'user_id' => $user->id,
                'cot_code' => $cotCode,
                'cot_status' => false,
                'cot_msg' => 'COT code generated',
                'tax_code' => $taxCode,
                'tax_status' => false,
                'tax_msg' => 'Tax code generated',
                'imf_code' => $imfCode,
                'imf_status' => false,
                'imf_msg' => 'IMF code generated',
                'allow_local' => false,
                'allow_international' => false,
            ]);

        } catch (Exception $e) {
            // Silently handle errors to prevent application crashes
        }
    }

    /**
     * Generate a unique code with the given prefix.
     */
    private function generateUniqueCode(string $prefix): string
    {
        $maxAttempts = 10;
        $attempts = 0;

        do {
            $code = $prefix . strtoupper(Str::random(6));
            $exists = Code::where('cot_code', $code)
                ->orWhere('tax_code', $code)
                ->orWhere('imf_code', $code)
                ->exists();
            
            $attempts++;
        } while ($exists && $attempts < $maxAttempts);

        if ($attempts >= $maxAttempts) {
            throw new Exception("Unable to generate unique code for prefix {$prefix}");
        }

        return $code;
    }
}
