<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use OwenIt\Auditing\Contracts\Auditable;
use OwenIt\Auditing\Auditable as AuditableTrait;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;
use Akaunting\Money\Money;

class VirtualCard extends Model implements Auditable
{
    use HasFactory, SoftDeletes, AuditableTrait, LogsActivity;

    /**
     * The attributes that are mass assignable.
     */
    protected $fillable = [
        'user_id',
        'account_id',
        'card_number',
        'card_type',
        'card_brand',
        'card_name',
        'cvv',
        'expiry_date',
        'currency',
        'credit_limit',
        'current_balance',
        'daily_limit',
        'monthly_limit',
        'status',
        'activated_at',
        'online_payments',
        'atm_withdrawals',
        'international_usage',
        'total_transactions',
        'total_spent',
        'last_used_at'
    ];

    /**
     * The attributes that should be cast.
     */
    protected $casts = [
        'expiry_date' => 'date',
        'credit_limit' => 'decimal:4',
        'current_balance' => 'decimal:4',
        'daily_limit' => 'decimal:4',
        'monthly_limit' => 'decimal:4',
        'online_payments' => 'boolean',
        'atm_withdrawals' => 'boolean',
        'international_usage' => 'boolean',
        'total_transactions' => 'integer',
        'total_spent' => 'decimal:4',
        'activated_at' => 'datetime',
        'last_used_at' => 'datetime',
    ];

    /**
     * The attributes that should be hidden for serialization.
     */
    protected $hidden = [
        'id',
        'user_id',
        'account_id',
        'cvv'
    ];

    /**
     * The attributes that should be appended.
     */
    protected $appends = [
        'masked_card_number',
        'formatted_credit_limit',
        'formatted_current_balance',
        'formatted_daily_limit',
        'formatted_monthly_limit',
        'card_status_label',
        'is_active',
        'is_expired',
        'is_blocked',
        'days_until_expiry',
        'can_make_payment',
        'daily_limit_remaining',
        'monthly_limit_remaining'
    ];

    /**
     * Get the user that owns the virtual card.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the account associated with the virtual card.
     */
    public function account(): BelongsTo
    {
        return $this->belongsTo(Account::class);
    }

    /**
     * Get the transactions for this virtual card.
     */
    public function transactions()
    {
        return Transaction::where(function($query) {
            $query->where('from_account_id', $this->account_id)
                  ->orWhere('to_account_id', $this->account_id);
        });
    }

    /**
     * Get the masked card number.
     */
    public function getMaskedCardNumberAttribute(): string
    {
        $number = $this->card_number;
        if (strlen($number) < 4) {
            return $number;
        }
        
        return substr($number, 0, 4) . str_repeat('*', strlen($number) - 8) . substr($number, -4);
    }

    /**
     * Get the formatted credit limit.
     */
    public function getFormattedCreditLimitAttribute(): string
    {
        if (!$this->credit_limit) {
            return 'N/A';
        }
        
        return \App\Helpers\CurrencyHelper::getSymbol($this->currency) . number_format($this->credit_limit, 2);
    }

    /**
     * Get the formatted current balance.
     */
    public function getFormattedCurrentBalanceAttribute(): string
    {
        return \App\Helpers\CurrencyHelper::getSymbol($this->currency) . number_format($this->current_balance, 2);
    }

    /**
     * Get the formatted daily limit.
     */
    public function getFormattedDailyLimitAttribute(): string
    {
        if (!$this->daily_limit) {
            return 'No Limit';
        }
        
        return \App\Helpers\CurrencyHelper::getSymbol($this->currency) . number_format($this->daily_limit, 2);
    }

    /**
     * Get the formatted monthly limit.
     */
    public function getFormattedMonthlyLimitAttribute(): string
    {
        if (!$this->monthly_limit) {
            return 'No Limit';
        }
        
        return \App\Helpers\CurrencyHelper::getSymbol($this->currency) . number_format($this->monthly_limit, 2);
    }

    /**
     * Get the card status label.
     */
    public function getCardStatusLabelAttribute(): string
    {
        return match($this->status) {
            'active' => 'Active',
            'inactive' => 'Inactive',
            'blocked' => 'Blocked',
            default => 'Unknown'
        };
    }

    /**
     * Check if card is active.
     */
    public function getIsActiveAttribute(): bool
    {
        return $this->status === 'active';
    }

    /**
     * Check if card is expired.
     */
    public function getIsExpiredAttribute(): bool
    {
        return $this->expiry_date && now()->isAfter($this->expiry_date);
    }

    /**
     * Check if card is blocked.
     */
    public function getIsBlockedAttribute(): bool
    {
        return $this->status === 'blocked';
    }

    /**
     * Get days until expiry.
     */
    public function getDaysUntilExpiryAttribute(): ?int
    {
        if (!$this->expiry_date) {
            return null;
        }
        
        return now()->diffInDays($this->expiry_date, false);
    }

    /**
     * Check if card can make payment.
     */
    public function getCanMakePaymentAttribute(): bool
    {
        return $this->is_active && !$this->is_expired && !$this->is_blocked;
    }

    /**
     * Get remaining daily limit.
     */
    public function getDailyLimitRemainingAttribute(): ?float
    {
        if (!$this->daily_limit) {
            return null;
        }

        $todayTransactions = $this->transactions()
            ->whereDate('created_at', today())
            ->sum('amount');

        return max(0, $this->daily_limit - $todayTransactions);
    }

    /**
     * Get remaining monthly limit.
     */
    public function getMonthlyLimitRemainingAttribute(): ?float
    {
        if (!$this->monthly_limit) {
            return null;
        }

        $monthlyTransactions = $this->transactions()
            ->whereMonth('created_at', now()->month)
            ->whereYear('created_at', now()->year)
            ->sum('amount');

        return max(0, $this->monthly_limit - $monthlyTransactions);
    }

    /**
     * Scope for active cards.
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    /**
     * Scope for inactive cards.
     */
    public function scopeInactive($query)
    {
        return $query->where('status', 'inactive');
    }

    /**
     * Scope for blocked cards.
     */
    public function scopeBlocked($query)
    {
        return $query->where('status', 'blocked');
    }

    /**
     * Scope for cards by type.
     */
    public function scopeOfType($query, $type)
    {
        return $query->where('card_type', $type);
    }

    /**
     * Scope for cards by brand.
     */
    public function scopeOfBrand($query, $brand)
    {
        return $query->where('card_brand', $brand);
    }

    /**
     * Scope for cards by currency.
     */
    public function scopeInCurrency($query, $currency)
    {
        return $query->where('currency', $currency);
    }

    /**
     * Scope for expired cards.
     */
    public function scopeExpired($query)
    {
        return $query->where('expiry_date', '<', now());
    }

    /**
     * Scope for expiring soon cards.
     */
    public function scopeExpiringSoon($query, $days = 30)
    {
        return $query->whereBetween('expiry_date', [now(), now()->addDays($days)]);
    }

    /**
     * Scope for cards with online payments enabled.
     */
    public function scopeOnlinePaymentsEnabled($query)
    {
        return $query->where('online_payments', true);
    }

    /**
     * Scope for cards with ATM withdrawals enabled.
     */
    public function scopeAtmWithdrawalsEnabled($query)
    {
        return $query->where('atm_withdrawals', true);
    }

    /**
     * Scope for international usage enabled.
     */
    public function scopeInternationalUsageEnabled($query)
    {
        return $query->where('international_usage', true);
    }

    /**
     * Activate the card.
     */
    public function activate(): bool
    {
        $this->status = 'active';
        $this->activated_at = now();
        
        if ($this->save()) {
            // Log the activation
            activity()
                ->performedOn($this)
                ->log("Virtual card {$this->masked_card_number} activated");
            
            return true;
        }

        return false;
    }

    /**
     * Deactivate the card.
     */
    public function deactivate(): bool
    {
        $this->status = 'inactive';
        
        if ($this->save()) {
            // Log the deactivation
            activity()
                ->performedOn($this)
                ->log("Virtual card {$this->masked_card_number} deactivated");
            
            return true;
        }

        return false;
    }

    /**
     * Block the card.
     */
    public function block(string $reason = null): bool
    {
        $this->status = 'blocked';
        
        if ($this->save()) {
            // Log the blocking
            activity()
                ->performedOn($this)
                ->log("Virtual card {$this->masked_card_number} blocked: {$reason}");
            
            return true;
        }

        return false;
    }

    /**
     * Update card usage statistics.
     */
    public function updateUsageStatistics(float $amount): bool
    {
        $this->total_transactions++;
        $this->total_spent += $amount;
        $this->last_used_at = now();
        
        return $this->save();
    }

    /**
     * Check if card can process transaction.
     */
    public function canProcessTransaction(float $amount, string $type = 'payment'): bool
    {
        if (!$this->can_make_payment) {
            return false;
        }

        // Check daily limit
        if ($this->daily_limit && $this->daily_limit_remaining < $amount) {
            return false;
        }

        // Check monthly limit
        if ($this->monthly_limit && $this->monthly_limit_remaining < $amount) {
            return false;
        }

        // Check if online payments are enabled for online transactions
        if ($type === 'online_payment' && !$this->online_payments) {
            return false;
        }

        // Check if ATM withdrawals are enabled for ATM transactions
        if ($type === 'atm_withdrawal' && !$this->atm_withdrawals) {
            return false;
        }

        return true;
    }

    /**
     * Get card usage summary.
     */
    public function getUsageSummary(string $period = 'month'): array
    {
        $query = $this->transactions();

        switch ($period) {
            case 'week':
                $query->whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()]);
                break;
            case 'month':
                $query->whereMonth('created_at', now()->month)
                    ->whereYear('created_at', now()->year);
                break;
            case 'year':
                $query->whereYear('created_at', now()->year);
                break;
        }

        return [
            'count' => $query->count(),
            'total_amount' => $query->sum('amount'),
            'average_amount' => $query->avg('amount'),
            'last_transaction' => $query->latest()->first()
        ];
    }

    /**
     * Get validation rules.
     */
    public static function getValidationRules(): array
    {
        return [
            'card_type' => 'required|in:debit,credit,prepaid',
            'card_brand' => 'required|in:visa,mastercard,amex',
            'card_name' => 'nullable|string|max:255',
            'expiry_date' => 'required|date|after:today',
            'currency' => 'required|string|size:3',
            'credit_limit' => 'nullable|numeric|min:0',
            'daily_limit' => 'nullable|numeric|min:0',
            'monthly_limit' => 'nullable|numeric|min:0',
            'online_payments' => 'boolean',
            'atm_withdrawals' => 'boolean',
            'international_usage' => 'boolean',
        ];
    }

    /**
     * Activity log options.
     */
    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly([
                'card_number', 'card_type', 'card_brand', 'card_name',
                'status', 'daily_limit', 'monthly_limit'
            ])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }

    /**
     * Audit configuration.
     */
    protected $auditInclude = [
        'card_number', 'card_type', 'card_brand', 'card_name',
        'status', 'daily_limit', 'monthly_limit'
    ];

    protected $auditEvents = [
        'created', 'updated', 'deleted', 'restored'
    ];
}
