<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use OwenIt\Auditing\Contracts\Auditable;
use OwenIt\Auditing\Auditable as AuditableTrait;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;

class UserProfile extends Model implements Auditable
{
    use HasFactory, SoftDeletes, AuditableTrait, LogsActivity;

    /**
     * The attributes that are mass assignable.
     */
    protected $fillable = [
        'user_id',
        'first_name',
        'last_name',
        'middle_name',
        'date_of_birth',
        'gender',
        'marital_status',
        'occupation',
        'annual_income',
        'address_line_1',
        'city',
        'state_province',
        'postal_code',
        'country',
        'phone_number',
        'profile_image',
        'profile_image_disk',
        'emergency_contact_name',
        'emergency_contact_phone',
        'emergency_contact_relationship',
        'preferred_language',
        'timezone',
        'kyc_status',
        'kyc_verified_at',
        'aml_status',
        'compliance_status',
        'notes',
        'metadata'
    ];

    /**
     * The attributes that should be cast.
     */
    protected $casts = [
        'date_of_birth' => 'date',
        'annual_income' => 'decimal:2',
        'kyc_verified_at' => 'datetime',
        'metadata' => 'array'
    ];

    /**
     * The attributes that should be hidden for serialization.
     */
    protected $hidden = [
        'id',
        'user_id'
    ];

    /**
     * The attributes that should be appended.
     */
    protected $appends = [
        'full_name',
        'full_address',
        'age',
        'profile_image_url',
        'kyc_status_label',
        'aml_status_label',
        'compliance_status_label',
        'is_kyc_complete',
        'is_aml_clear',
        'is_compliant'
    ];

    /**
     * Get the user that owns the profile.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the full name of the user.
     */
    public function getFullNameAttribute(): string
    {
        $name = trim($this->first_name . ' ' . $this->last_name);
        if ($this->middle_name) {
            $name = trim($this->first_name . ' ' . $this->middle_name . ' ' . $this->last_name);
        }
        return $name;
    }

    /**
     * Get the full address of the user.
     */
    public function getFullAddressAttribute(): string
    {
        $address = $this->address_line_1;
        $address .= ', ' . $this->city;
        if ($this->state_province) {
            $address .= ', ' . $this->state_province;
        }
        $address .= ' ' . $this->postal_code . ', ' . $this->country;
        return $address;
    }

    /**
     * Get the age of the user.
     */
    public function getAgeAttribute(): ?int
    {
        if (!$this->date_of_birth) {
            return null;
        }
        
        return $this->date_of_birth->age;
    }

    /**
     * Get the KYC status label.
     */
    public function getKycStatusLabelAttribute(): string
    {
        return match($this->kyc_status) {
            'pending' => 'Pending Verification',
            'in_progress' => 'Verification In Progress',
            'verified' => 'Verified',
            'rejected' => 'Rejected',
            'expired' => 'Expired',
            'suspended' => 'Suspended',
            default => 'Unknown'
        };
    }

    /**
     * Get the AML status label.
     */
    public function getAmlStatusLabelAttribute(): string
    {
        return match($this->aml_status) {
            'clear' => 'Clear',
            'pending' => 'Pending Review',
            'flagged' => 'Flagged for Review',
            'blocked' => 'Blocked',
            default => 'Unknown'
        };
    }

    /**
     * Get the compliance status label.
     */
    public function getComplianceStatusLabelAttribute(): string
    {
        return match($this->compliance_status) {
            'compliant' => 'Compliant',
            'non_compliant' => 'Non-Compliant',
            'pending_review' => 'Pending Review',
            'under_investigation' => 'Under Investigation',
            default => 'Unknown'
        };
    }

    /**
     * Check if KYC is complete.
     */
    public function getIsKycCompleteAttribute(): bool
    {
        return $this->kyc_status === 'verified';
    }

    /**
     * Get the profile image URL.
     */
    public function getProfileImageUrlAttribute(): ?string
    {
        if (!$this->profile_image) {
            return null;
        }
        
        return asset('public/' . $this->profile_image);
    }

    /**
     * Check if AML is clear.
     */
    public function getIsAmlClearAttribute(): bool
    {
        return $this->aml_status === 'clear';
    }

    /**
     * Check if user is compliant.
     */
    public function getIsCompliantAttribute(): bool
    {
        return $this->compliance_status === 'compliant';
    }

    /**
     * Scope for verified KYC profiles.
     */
    public function scopeKycVerified($query)
    {
        return $query->where('kyc_status', 'verified');
    }

    /**
     * Scope for pending KYC profiles.
     */
    public function scopeKycPending($query)
    {
        return $query->whereIn('kyc_status', ['pending', 'in_progress']);
    }

    /**
     * Scope for AML clear profiles.
     */
    public function scopeAmlClear($query)
    {
        return $query->where('aml_status', 'clear');
    }

    /**
     * Scope for compliant profiles.
     */
    public function scopeCompliant($query)
    {
        return $query->where('compliance_status', 'compliant');
    }



    /**
     * Mark KYC as verified.
     */
    public function markKycAsVerified(): bool
    {
        $this->kyc_status = 'verified';
        $this->kyc_verified_at = now();
        
        if ($this->save()) {
            // Log the KYC verification
            activity()
                ->performedOn($this)
                ->log("KYC verified for user {$this->full_name}");
            
            return true;
        }

        return false;
    }

    /**
     * Mark KYC as rejected.
     */
    public function markKycAsRejected(string $reason = null): bool
    {
        $this->kyc_status = 'rejected';
        
        if ($this->save()) {
            // Log the KYC rejection
            activity()
                ->performedOn($this)
                ->log("KYC rejected for user {$this->full_name}: {$reason}");
            
            return true;
        }

        return false;
    }

    /**
     * Mark AML as clear.
     */
    public function markAmlAsClear(): bool
    {
        $this->aml_status = 'clear';
        
        if ($this->save()) {
            // Log the AML clearance
            activity()
                ->performedOn($this)
                ->log("AML cleared for user {$this->full_name}");
            
            return true;
        }

        return false;
    }

    /**
     * Mark AML as flagged.
     */
    public function markAmlAsFlagged(string $reason = null): bool
    {
        $this->aml_status = 'flagged';
        
        if ($this->save()) {
            // Log the AML flag
            activity()
                ->performedOn($this)
                ->log("AML flagged for user {$this->full_name}: {$reason}");
            
            return true;
        }

        return false;
    }

    /**
     * Mark as compliant.
     */
    public function markAsCompliant(): bool
    {
        $this->compliance_status = 'compliant';
        
        if ($this->save()) {
            // Log the compliance status
            activity()
                ->performedOn($this)
                ->log("User {$this->full_name} marked as compliant");
            
            return true;
        }

        return false;
    }

    /**
     * Get profile completeness percentage.
     */
    public function getCompletenessPercentage(): int
    {
        $requiredFields = [
            'first_name', 'last_name', 'date_of_birth',
            'address_line_1', 'city', 'postal_code', 'country',
            'phone_number', 'occupation'
        ];

        $filledFields = 0;
        foreach ($requiredFields as $field) {
            if (!empty($this->$field)) {
                $filledFields++;
            }
        }

        return round(($filledFields / count($requiredFields)) * 100);
    }

    /**
     * Check if profile is complete for banking.
     */
    public function isCompleteForBanking(): bool
    {
        return $this->getCompletenessPercentage() >= 80;
    }

    /**
     * Get validation rules.
     */
    public static function getValidationRules(): array
    {
        return [
            'first_name' => 'required|string|max:100',
            'last_name' => 'required|string|max:100',
            'middle_name' => 'nullable|string|max:100',
            'date_of_birth' => 'required|date|before:today',
            'gender' => 'nullable|in:male,female,other,prefer_not_to_say',
            'marital_status' => 'nullable|in:single,married,divorced,widowed,separated',
            'occupation' => 'required|string|max:255',
            'annual_income' => 'nullable|numeric|min:0',
            'address_line_1' => 'required|string|max:255',
            'city' => 'required|string|max:100',
            'state_province' => 'nullable|string|max:100',
            'postal_code' => 'required|string|max:20',
            'country' => 'required|string|max:100',
            'phone_number' => 'required|string|max:20',
            'emergency_contact_name' => 'nullable|string|max:255',
            'emergency_contact_phone' => 'nullable|string|max:20',
            'emergency_contact_relationship' => 'nullable|string|max:100',
            'preferred_language' => 'nullable|string|max:10',
            'timezone' => 'nullable|string|max:50',
            'notes' => 'nullable|string|max:1000'
        ];
    }

    /**
     * Activity log options.
     */
    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly([
                'first_name', 'last_name', 'date_of_birth',
                'occupation', 'country', 'kyc_status', 'aml_status', 'compliance_status'
            ])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }

    /**
     * Audit configuration.
     */
    protected $auditInclude = [
        'first_name', 'last_name', 'date_of_birth',
        'occupation', 'country', 'kyc_status', 'aml_status', 'compliance_status'
    ];

    protected $auditEvents = [
        'created', 'updated', 'deleted', 'restored'
    ];
}
