<?php

namespace App\Models;

use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Str;
use Spatie\Permission\Traits\HasRoles;
use OwenIt\Auditing\Contracts\Auditable;
use OwenIt\Auditing\Auditable as AuditableTrait;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;
use Spatie\LaravelPasskeys\Models\Concerns\HasPasskeys;
use Spatie\LaravelPasskeys\Models\Concerns\InteractsWithPasskeys;
use Lab404\Impersonate\Models\Impersonate;

class User extends Authenticatable implements MustVerifyEmail, Auditable, HasPasskeys
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable, HasRoles, AuditableTrait, LogsActivity, InteractsWithPasskeys, Impersonate;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'email',
        'email_verified_at',
        'username',
        'password',
        'password_changed_at',
        'customer_id',
        'customer_type',
        'pin',
        'customer_status',
        'account_activated_at',
        'last_login_at',
        'last_activity_at',
        'last_login_ip',
        'email_verification_code',
        'email_verification_code_expires_at',
        'email_verification_attempts',
        'last_email_verification_attempt',
        'password_reset_code',
        'password_reset_code_expires_at',
        'password_reset_attempts',
        'last_password_reset_attempt',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
        'username',
        'customer_id',
        'pin',
    ];

    /**
     * The attributes that should be appended.
     */
    protected $appends = [
        'customer_status_label',
        'is_business_customer',
        'is_premium_customer',
        'is_account_active',
        'is_account_suspended',
        'is_account_pending',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'account_activated_at' => 'datetime',
            'last_login_at' => 'datetime',
            'last_activity_at' => 'datetime',
            'password_changed_at' => 'datetime',
            'email_verification_code_expires_at' => 'datetime',
            'last_email_verification_attempt' => 'datetime',
            'password_reset_code_expires_at' => 'datetime',
            'last_password_reset_attempt' => 'datetime',
        ];
    }

    /**
     * Boot method to auto-generate customer_id
     */
    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($user) {
            if (empty($user->customer_id)) {
                $user->customer_id = static::generateCustomerId();
            }
        });
    }

    /**
     * Generate a unique 8-digit customer ID
     */
    protected static function generateCustomerId(): string
    {
        do {
            $customerId = str_pad(random_int(10000000, 99999999), 8, '0', STR_PAD_LEFT);
        } while (static::where('customer_id', $customerId)->exists());
        
        return $customerId;
    }

    /**
     * Get the user's initials
     */
    public function initials(): string
    {
        return Str::of($this->name)
            ->explode(' ')
            ->take(2)
            ->map(fn ($word) => Str::substr($word, 0, 1))
            ->implode('');
    }

    /**
     * Check if user is a super admin
     */
    public function isSuperAdmin(): bool
    {
        return $this->hasRole('super-admin');
    }

    /**
     * Check if user is an admin
     */
    public function isAdmin(): bool
    {
        return $this->hasRole('admin') || $this->hasRole('super-admin');
    }

    /**
     * Check if user is a customer
     */
    public function isCustomer(): bool
    {
        return $this->hasRole('customer');
    }

    /**
     * Get the user's primary role
     */
    public function getPrimaryRoleAttribute(): string
    {
        return $this->roles->first()->name ?? 'No Role';
    }

    /**
     * Get the user's profile.
     */
    public function profile()
    {
        return $this->hasOne(UserProfile::class);
    }

    /**
     * Get the user's accounts.
     */
    public function accounts()
    {
        return $this->hasMany(Account::class);
    }

    /**
     * Get the user's master account.
     */
    public function masterAccount()
    {
        return $this->hasOne(Account::class)->where('is_master_account', true);
    }

    /**
     * Get the user's transactions.
     */
    public function transactions()
    {
        return $this->hasMany(Transaction::class);
    }

    /**
     * Get the user's beneficiaries.
     */
    public function beneficiaries()
    {
        return $this->hasMany(Beneficiary::class);
    }

    public function virtualCards()
    {
        return $this->hasMany(VirtualCard::class);
    }

    public function notifications()
    {
        return $this->hasMany(Notification::class);
    }

    /**
     * Get the user's two-factor authentication.
     */
    public function twoFactorAuth()
    {
        return $this->hasOne(TwoFactorAuth::class);
    }

    public function kycDocuments()
    {
        return $this->hasMany(KycDocument::class);
    }

    public function complianceLogs()
    {
        return $this->hasMany(ComplianceLog::class);
    }

    public function code()
    {
        return $this->hasOne(Code::class);
    }

    /**
     * Get the customer status label.
     */
    public function getCustomerStatusLabelAttribute(): string
    {
        if (!$this->customer_status) {
            return 'Unknown';
        }
        
        return match($this->customer_status) {
            'active' => 'Active',
            'inactive' => 'Inactive',
            'suspended' => 'Suspended',
            'pending_verification' => 'Pending Verification',
            'transaction_declined' => 'Transaction Declined',
            'closed' => 'Closed',
            'blocked' => 'Blocked',
            default => 'Unknown'
        };
    }

    /**
     * Check if user is a business customer.
     */
    public function getIsBusinessCustomerAttribute(): bool
    {
        return in_array($this->customer_type, ['business', 'corporate']);
    }

    /**
     * Check if user is a premium customer.
     */
    public function getIsPremiumCustomerAttribute(): bool
    {
        return $this->customer_type === 'premium';
    }

    /**
     * Check if account is active.
     */
    public function getIsAccountActiveAttribute(): bool
    {
        return $this->customer_status === 'active';
    }

    /**
     * Check if account is suspended.
     */
    public function getIsAccountSuspendedAttribute(): bool
    {
        return $this->customer_status === 'suspended';
    }

    /**
     * Check if account is pending verification.
     */
    public function getIsAccountPendingAttribute(): bool
    {
        return $this->customer_status === 'pending_verification';
    }

    /**
     * Update last activity timestamp.
     */
    public function updateLastActivity(): bool
    {
        $this->last_activity_at = now();
        return $this->save();
    }

    /**
     * Update last login timestamp.
     */
    public function updateLastLogin(): bool
    {
        $this->last_login_at = now();
        $this->last_activity_at = now();
        return $this->save();
    }

    /**
     * Activate user account.
     */
    public function activateAccount(): bool
    {
        $this->customer_status = 'active';
        $this->account_activated_at = now();
        
        if ($this->save()) {
            // Log the activation
            activity()
                ->performedOn($this)
                ->log("User account activated for {$this->name}");
            
            return true;
        }

        return false;
    }

    /**
     * Suspend user account.
     */
    public function suspendAccount(string $reason = null): bool
    {
        $this->customer_status = 'suspended';
        
        if ($this->save()) {
            // Log the suspension
            activity()
                ->performedOn($this)
                ->log("User account suspended for {$this->name}: {$reason}");
            
            return true;
        }

        return false;
    }

    /**
     * Get user's total balance across all accounts.
     */
    public function getTotalBalance(): float
    {
        return $this->accounts()->sum('balance');
    }

    /**
     * Get user's total balance in specific currency.
     */
    public function getTotalBalanceInCurrency(string $currency): float
    {
        if ($currency === 'primary') {
            $primaryAccount = $this->accounts()->where('is_master_account', true)->first();
            return $primaryAccount ? $primaryAccount->balance : 0;
        }
        
        return $this->accounts()
            ->where('currency', $currency)
            ->sum('balance');
    }

    public function getTotalBalanceAttribute()
    {
        $primaryAccount = $this->accounts()->where('is_master_account', true)->first();
        return $primaryAccount ? $primaryAccount->balance : 0;
    }

    /**
     * Activity log options
     */
    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly([
                'name', 'email', 'customer_type', 'customer_status'
            ])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs()
            ->logExcept(['pin']); // Don't log sensitive PIN data
    }

    /**
     * Audit configuration
     */
    protected $auditInclude = [
        'name', 'email', 'customer_type', 'customer_status'
    ];

    protected $auditExclude = [
        'pin' // Exclude sensitive PIN from audits
    ];

    protected $auditEvents = [
        'created',
        'updated',
        'deleted',
        'restored'
    ];

    /**
     * Check if the user can impersonate other users
     */
    public function canImpersonate(): bool
    {
        // Only super-admin and admin roles can impersonate
        return $this->hasRole(['super-admin', 'admin']);
    }

    /**
     * Check if the user can be impersonated
     */
    public function canBeImpersonated(): bool
    {
        // Cannot impersonate super-admin users
        return !$this->hasRole('super-admin');
    }

    /**
     * Send the email verification notification.
     */
    public function sendEmailVerificationNotification()
    {
        $this->notify(new \App\Notifications\VerifyEmailNotification);
    }

    /**
     * Send the password reset notification.
     */
    public function sendPasswordResetNotification($token)
    {
        $this->notify(new \App\Notifications\ResetPasswordNotification($token));
    }

    /**
     * Generate email verification code.
     */
    public function generateEmailVerificationCode(): string
    {
        $code = str_pad(random_int(0, 999999), 6, '0', STR_PAD_LEFT);
        $this->email_verification_code = $code;
        $this->email_verification_code_expires_at = now()->addMinutes(10);
        $this->email_verification_attempts = 0;
        $this->save();
        
        return $code;
    }

    /**
     * Verify email verification code.
     */
    public function verifyEmailCode(string $code): bool
    {
        if (!$this->email_verification_code || 
            !$this->email_verification_code_expires_at || 
            now()->isAfter($this->email_verification_code_expires_at)) {
            return false;
        }

        $isValid = $this->email_verification_code === $code;
        
        if ($isValid) {
            $this->email_verified_at = now();
            $this->email_verification_code = null;
            $this->email_verification_code_expires_at = null;
            $this->email_verification_attempts = 0;
            $this->save();
        } else {
            $this->email_verification_attempts++;
            $this->last_email_verification_attempt = now();
            $this->save();
        }

        return $isValid;
    }

    /**
     * Generate password reset code.
     */
    public function generatePasswordResetCode(): string
    {
        $code = str_pad(random_int(0, 999999), 6, '0', STR_PAD_LEFT);
        $this->password_reset_code = $code;
        $this->password_reset_code_expires_at = now()->addMinutes(15);
        $this->password_reset_attempts = 0;
        $this->save();
        
        return $code;
    }

    /**
     * Verify password reset code.
     */
    public function verifyPasswordResetCode(string $code): bool
    {
        if (!$this->password_reset_code || 
            !$this->password_reset_code_expires_at || 
            now()->isAfter($this->password_reset_code_expires_at)) {
            return false;
        }

        $isValid = $this->password_reset_code === $code;
        
        if (!$isValid) {
            $this->password_reset_attempts++;
            $this->last_password_reset_attempt = now();
            $this->save();
        }

        return $isValid;
    }

    /**
     * Clear password reset code.
     */
    public function clearPasswordResetCode(): void
    {
        $this->password_reset_code = null;
        $this->password_reset_code_expires_at = null;
        $this->password_reset_attempts = 0;
        $this->save();
    }

    /**
     * Check if email verification code is expired.
     */
    public function isEmailVerificationCodeExpired(): bool
    {
        return !$this->email_verification_code_expires_at || 
               now()->isAfter($this->email_verification_code_expires_at);
    }

    /**
     * Check if password reset code is expired.
     */
    public function isPasswordResetCodeExpired(): bool
    {
        return !$this->password_reset_code_expires_at || 
               now()->isAfter($this->password_reset_code_expires_at);
    }
}
