<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use OwenIt\Auditing\Contracts\Auditable;
use OwenIt\Auditing\Auditable as AuditableTrait;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;

class TwoFactorAuth extends Model implements Auditable
{
    use HasFactory, SoftDeletes, AuditableTrait, LogsActivity;

    /**
     * The table associated with the model.
     */
    protected $table = 'two_factor_auth';

    /**
     * The attributes that are mass assignable.
     */
    protected $fillable = [
        'user_id',
        'enabled',
        'method',
        'secret_key',
        'phone_number',
        'email',
        'verification_code',
        'verification_code_expires_at',
        'backup_codes',
        'totp_counter',
        'passkeys',
        'passkey_name',
        'passkey_credential_id',
        'passkey_public_key',
        'last_used_at',
        'notes',
        'metadata'
    ];

    /**
     * The attributes that should be cast.
     */
    protected $casts = [
        'enabled' => 'boolean',
        'verification_code_expires_at' => 'datetime',
        'backup_codes' => 'array',
        'totp_counter' => 'integer',
        'passkeys' => 'array',
        'last_used_at' => 'datetime',
        'metadata' => 'array'
    ];

    /**
     * The attributes that should be hidden for serialization.
     */
    protected $hidden = [
        'id',
        'user_id',
        'secret_key',
        'verification_code',
        'backup_codes',
        'passkey_credential_id',
        'passkey_public_key'
    ];

    /**
     * The attributes that should be appended.
     */
    protected $appends = [
        'method_label',
        'is_expired',
        'is_active',
        'has_backup_codes',
        'backup_codes_count'
    ];

    /**
     * Get the user that owns the 2FA configuration.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get method label.
     */
    public function getMethodLabelAttribute(): string
    {
        return match($this->method) {
            'totp' => 'Authenticator App (TOTP)',
            'email' => 'Email Verification',
            'passkey' => 'Passkey (FIDO2)',
            default => 'Unknown'
        };
    }

    /**
     * Check if verification code is expired.
     */
    public function getIsExpiredAttribute(): bool
    {
        if (!$this->verification_code_expires_at) {
            return true;
        }
        
        return now()->isAfter($this->verification_code_expires_at);
    }

    /**
     * Check if 2FA is active.
     */
    public function getIsActiveAttribute(): bool
    {
        return $this->enabled && $this->method;
    }

    /**
     * Check if backup codes exist.
     */
    public function getHasBackupCodesAttribute(): bool
    {
        return !empty($this->backup_codes) && count($this->backup_codes) > 0;
    }

    /**
     * Get backup codes count.
     */
    public function getBackupCodesCountAttribute(): int
    {
        return $this->backup_codes ? count($this->backup_codes) : 0;
    }

    /**
     * Scope for enabled 2FA.
     */
    public function scopeEnabled($query)
    {
        return $query->where('enabled', true);
    }

    /**
     * Scope for disabled 2FA.
     */
    public function scopeDisabled($query)
    {
        return $query->where('enabled', false);
    }

    /**
     * Scope for 2FA by method.
     */
    public function scopeByMethod($query, $method)
    {
        return $query->where('method', $method);
    }

    /**
     * Enable 2FA.
     */
    public function enable(): void
    {
        $this->enabled = true;
        $this->save();
    }

    /**
     * Disable 2FA.
     */
    public function disable(): void
    {
        $this->enabled = false;
        $this->save();
    }

    /**
     * Generate new backup codes.
     */
    public function generateBackupCodes(int $count = 8): array
    {
        $codes = [];
        for ($i = 0; $i < $count; $i++) {
            $codes[] = strtoupper(substr(md5(uniqid()), 0, 8));
        }
        
        $this->backup_codes = $codes;
        $this->save();
        
        return $codes;
    }

    /**
     * Verify backup code.
     */
    public function verifyBackupCode(string $code): bool
    {
        if (!$this->has_backup_codes) {
            return false;
        }
        
        $code = strtoupper($code);
        $codes = $this->backup_codes;
        
        if (in_array($code, $codes)) {
            // Remove used code
            $this->backup_codes = array_values(array_filter($codes, fn($c) => $c !== $code));
            $this->save();
            
            return true;
        }
        
        return false;
    }

    /**
     * Generate TOTP secret.
     */
    public function generateTotpSecret(): string
    {
        $secret = base32_encode(random_bytes(20));
        $this->secret_key = $secret;
        $this->method = 'totp';
        $this->totp_counter = 0;
        $this->save();
        
        return $secret;
    }

    /**
     * Generate email verification code.
     */
    public function generateEmailCode(): string
    {
        $code = str_pad(random_int(0, 999999), 6, '0', STR_PAD_LEFT);
        $this->verification_code = $code;
        $this->verification_code_expires_at = now()->addMinutes(10);
        $this->method = 'email';
        $this->save();
        
        return $code;
    }

    /**
     * Verify email code.
     */
    public function verifyEmailCode(string $code): bool
    {
        if ($this->is_expired) {
            return false;
        }
        
        return $this->verification_code === $code;
    }

    /**
     * Add passkey.
     */
    public function addPasskey(array $passkeyData): bool
    {
        $passkeys = $this->passkeys ?? [];
        $passkeys[] = $passkeyData;
        
        $this->passkeys = $passkeys;
        $this->method = 'passkey';
        $this->save();
        
        return true;
    }

    /**
     * Remove passkey.
     */
    public function removePasskey(string $credentialId): bool
    {
        $passkeys = $this->passkeys ?? [];
        $passkeys = array_filter($passkeys, fn($p) => $p['credential_id'] !== $credentialId);
        
        $this->passkeys = array_values($passkeys);
        
        if (empty($passkeys)) {
            $this->method = null;
        }
        
        $this->save();
        
        return true;
    }

    /**
     * Update last used timestamp.
     */
    public function updateLastUsed(): bool
    {
        $this->last_used_at = now();
        return $this->save();
    }

    /**
     * Check if 2FA is required for transaction.
     */
    public function isRequiredForTransaction(float $amount): bool
    {
        if (!$this->enabled) {
            return false;
        }
        
        // Always require 2FA for high-value transactions
        if ($amount >= 10000) {
            return true;
        }
        
        // Require 2FA for international transactions
        // This would be checked in the transaction context
        
        return false;
    }

    /**
     * Get 2FA setup instructions.
     */
    public function getSetupInstructions(): array
    {
        return match($this->method) {
            'totp' => [
                'title' => 'Set up Authenticator App',
                'steps' => [
                    'Install an authenticator app (Google Authenticator, Authy, etc.)',
                    'Scan the QR code or enter the secret key manually',
                    'Enter the 6-digit code from your app to verify'
                ]
            ],
            'email' => [
                'title' => 'Set up Email Verification',
                'steps' => [
                    'Enter your email address',
                    'Check your email for the verification code',
                    'Enter the 6-digit code to verify'
                ]
            ],
            'passkey' => [
                'title' => 'Set up Passkey',
                'steps' => [
                    'Use your device\'s biometric authentication',
                    'Or insert your security key',
                    'Follow the on-screen instructions'
                ]
            ],
            default => [
                'title' => 'Choose 2FA Method',
                'steps' => [
                    'Select your preferred 2FA method',
                    'Follow the setup instructions',
                    'Verify your setup is working'
                ]
            ]
        };
    }

    /**
     * Get validation rules.
     */
    public static function getValidationRules(): array
    {
        return [
            'method' => 'required|in:totp,email,authenticator_app,passkey',
            'phone_number' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:255',
            'notes' => 'nullable|string|max:1000'
        ];
    }

    /**
     * Activity log options.
     */
    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly([
                'enabled', 'method', 'phone_number', 'email'
            ])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }

    /**
     * Audit configuration.
     */
    protected $auditInclude = [
        'enabled', 'method', 'phone_number', 'email'
    ];

    protected $auditEvents = [
        'created', 'updated', 'deleted', 'restored'
    ];
}
