<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use OwenIt\Auditing\Contracts\Auditable;
use OwenIt\Auditing\Auditable as AuditableTrait;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;
use Akaunting\Money\Money;

class Transaction extends Model implements Auditable
{
    use HasFactory, SoftDeletes, AuditableTrait, LogsActivity;

    /**
     * The attributes that are mass assignable.
     */
    protected $fillable = [
        'transaction_id',
        'reference_number',
        'external_reference',
        'user_id',
        'from_account_id',
        'to_account_id',
        'beneficiary_id',
        'type',
        'category',
        'subcategory',
        'amount',
        'currency',
        'exchange_rate',
        'converted_amount',
        'fee_amount',
        'tax_amount',
        'net_amount',
        'description',
        'notes',
        'status',
        'transfer_method',
        'transfer_speed',
        'estimated_arrival',
        'actual_arrival',
        'external_reference',
        'verification_status',
        'processed_at',
        'completed_at',
        'reversed_at',
        'reversal_reason',
        'metadata',
        'ip_address',
        'user_agent',
        'device_id',
        'location',
        'two_factor_verified',
        'biometric_verified',
        'risk_level',
        'aml_flagged',
        'aml_notes',
        'fraud_flagged',
        'fraud_notes',
        'created_at'
    ];

    /**
     * The attributes that should be cast.
     */
    protected $casts = [
        'amount' => 'decimal:4',
        'exchange_rate' => 'decimal:6',
        'converted_amount' => 'decimal:4',
        'fee_amount' => 'decimal:4',
        'tax_amount' => 'decimal:4',
        'net_amount' => 'decimal:4',
        'processed_at' => 'datetime',
        'completed_at' => 'datetime',
        'reversed_at' => 'datetime',
        'estimated_arrival' => 'datetime',
        'actual_arrival' => 'datetime',
        'metadata' => 'array',
        'two_factor_verified' => 'boolean',
        'biometric_verified' => 'boolean',
        'aml_flagged' => 'boolean',
        'fraud_flagged' => 'boolean',
        'created_at' => 'datetime'
    ];

    /**
     * The attributes that should be hidden for serialization.
     */
    protected $hidden = [
        'id',
        'user_id',
        'account_id',
        'external_reference'
    ];

    /**
     * The attributes that should be appended.
     */
    protected $appends = [
        'formatted_amount',
        'formatted_total_amount',
        'transaction_status_label',
        'is_credit',
        'is_debit',
        'is_pending',
        'is_completed',
        'is_failed',
        'is_reversed'
    ];

    /**
     * Get the user that owns the transaction.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the account associated with the transaction.
     */
    public function account(): BelongsTo
    {
        return $this->belongsTo(Account::class);
    }

    /**
     * Get the from account associated with the transaction.
     */
    public function fromAccount(): BelongsTo
    {
        return $this->belongsTo(Account::class, 'from_account_id');
    }

    /**
     * Get the to account associated with the transaction.
     */
    public function toAccount(): BelongsTo
    {
        return $this->belongsTo(Account::class, 'to_account_id');
    }

    /**
     * Get the beneficiary associated with the transaction.
     */
    public function beneficiary(): BelongsTo
    {
        return $this->belongsTo(Beneficiary::class);
    }

    /**
     * Get the related transactions (for reversals, splits, etc.).
     */
    public function relatedTransactions(): HasMany
    {
        return $this->hasMany(Transaction::class, 'reference', 'transaction_id');
    }

    /**
     * Get the compliance logs for this transaction.
     */
    public function complianceLogs(): HasMany
    {
        return $this->hasMany(ComplianceLog::class);
    }

    /**
     * Get the formatted amount.
     */
    public function getFormattedAmountAttribute(): string
    {
        return Money::{$this->currency}($this->amount)->format();
    }

    /**
     * Get the formatted total amount.
     */
    public function getFormattedTotalAmountAttribute(): string
    {
        return Money::{$this->currency}($this->total_amount)->format();
    }

    /**
     * Get the transaction status label.
     */
    public function getTransactionStatusLabelAttribute(): string
    {
        return match($this->status) {
            'pending' => 'Pending',
            'processing' => 'Processing',
            'completed' => 'Completed',
            'failed' => 'Failed',
            'cancelled' => 'Cancelled',
            'reversed' => 'Reversed',
            'disputed' => 'Disputed',
            'suspended' => 'Suspended',
            default => 'Unknown'
        };
    }

    /**
     * Check if transaction is a credit.
     */
    public function getIsCreditAttribute(): bool
    {
        return in_array($this->type, ['deposit', 'credit', 'refund', 'interest']);
    }

    /**
     * Check if transaction is a debit.
     */
    public function getIsDebitAttribute(): bool
    {
        return in_array($this->type, ['withdrawal', 'debit', 'payment', 'transfer_out', 'fee']);
    }

    /**
     * Check if transaction is pending.
     */
    public function getIsPendingAttribute(): bool
    {
        return in_array($this->status, ['pending', 'processing']);
    }

    /**
     * Check if transaction is completed.
     */
    public function getIsCompletedAttribute(): bool
    {
        return $this->status === 'completed';
    }

    /**
     * Check if transaction is failed.
     */
    public function getIsFailedAttribute(): bool
    {
        return in_array($this->status, ['failed', 'cancelled']);
    }

    /**
     * Check if transaction is reversed.
     */
    public function getIsReversedAttribute(): bool
    {
        return $this->status === 'reversed';
    }

    /**
     * Scope for transactions by type.
     */
    public function scopeOfType($query, $type)
    {
        return $query->where('type', $type);
    }

    /**
     * Scope for transactions by category.
     */
    public function scopeInCategory($query, $category)
    {
        return $query->where('category', $category);
    }

    /**
     * Scope for transactions by status.
     */
    public function scopeWithStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope for transactions by date range.
     */
    public function scopeInDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('created_at', [$startDate, $endDate]);
    }

    /**
     * Scope for transactions by amount range.
     */
    public function scopeInAmountRange($query, $minAmount, $maxAmount)
    {
        return $query->whereBetween('amount', [$minAmount, $maxAmount]);
    }

    /**
     * Scope for credit transactions.
     */
    public function scopeCredits($query)
    {
        return $query->whereIn('type', ['deposit', 'credit', 'refund', 'interest']);
    }

    /**
     * Scope for debit transactions.
     */
    public function scopeDebits($query)
    {
        return $query->whereIn('type', ['withdrawal', 'debit', 'payment', 'transfer_out', 'fee']);
    }

    /**
     * Scope for pending transactions.
     */
    public function scopePending($query)
    {
        return $query->whereIn('status', ['pending', 'processing']);
    }

    /**
     * Scope for completed transactions.
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    /**
     * Scope for failed transactions.
     */
    public function scopeFailed($query)
    {
        return $query->whereIn('status', ['failed', 'cancelled']);
    }

    /**
     * Scope for international transactions.
     */
    public function scopeInternational($query)
    {
        return $query->where('is_international', true);
    }

    /**
     * Scope for recurring transactions.
     */
    public function scopeRecurring($query)
    {
        return $query->where('is_recurring', true);
    }

    /**
     * Get transaction summary for account.
     */
    public static function getAccountSummary(int $accountId, string $period = 'month'): array
    {
        $query = static::where('account_id', $accountId)
            ->where('status', 'completed');

        switch ($period) {
            case 'week':
                $query->whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()]);
                break;
            case 'month':
                $query->whereMonth('created_at', now()->month)
                    ->whereYear('created_at', now()->year);
                break;
            case 'year':
                $query->whereYear('created_at', now()->year);
                break;
        }

        $credits = $query->clone()->credits()->sum('amount');
        $debits = $query->clone()->debits()->sum('amount');
        $fees = $query->clone()->where('type', 'fee')->sum('amount');

        return [
            'credits' => $credits,
            'debits' => $debits,
            'fees' => $fees,
            'net' => $credits - $debits - $fees,
            'count' => $query->count()
        ];
    }

    /**
     * Process the transaction.
     */
    public function process(): bool
    {
        if ($this->status !== 'pending') {
            return false;
        }

        $this->status = 'processing';
        $this->processed_at = now();
        
        if ($this->save()) {
            // Log the processing
            activity()
                ->performedOn($this)
                ->log("Transaction {$this->transaction_id} processing started");
            
            return true;
        }

        return false;
    }

    /**
     * Complete the transaction.
     */
    public function complete(): bool
    {
        if ($this->status !== 'processing') {
            return false;
        }

        $this->status = 'completed';
        $this->completed_at = now();
        
        if ($this->save()) {
            // Update account balance
            if ($this->fromAccount) {
                $this->fromAccount->updateBalance($this->amount, $this->is_credit ? 'credit' : 'debit');
            }

            // Check for compliance flags
            $this->checkComplianceFlags();

            // Log the completion
            activity()
                ->performedOn($this)
                ->log("Transaction {$this->transaction_id} completed successfully");
            
            return true;
        }

        return false;
    }

    /**
     * Reverse the transaction.
     */
    public function reverse(string $reason = null): bool
    {
        if (!in_array($this->status, ['completed', 'processing'])) {
            return false;
        }

        $this->status = 'reversed';
        $this->reversed_at = now();
        $this->reversal_reason = $reason;
        
        if ($this->save()) {
            // Reverse account balance if completed
            if ($this->status === 'completed' && $this->account) {
                $this->account->updateBalance($this->amount, $this->is_credit ? 'debit' : 'credit');
            }

            // Log the reversal
            activity()
                ->performedOn($this)
                ->log("Transaction {$this->transaction_id} reversed: {$reason}");
            
            return true;
        }

        return false;
    }

    /**
     * Check for compliance flags and create logs if needed.
     */
    public function checkComplianceFlags(): void
    {
        $complianceService = app(\App\Services\ComplianceService::class);
        
        // Run AML monitoring (new comprehensive monitoring)
        $complianceService->monitorTransaction($this);
        
        // Legacy compliance checks (kept for backward compatibility)
        // High value transaction flag
        if ($this->amount >= 10000) {
            $complianceService->flagTransaction(
                $this,
                "High value transaction: {$this->formatted_amount}",
                'high'
            );
        }
        
        // Round number flag (potential structuring)
        if ($this->amount % 1000 === 0 && $this->amount >= 5000) {
            $complianceService->flagTransaction(
                $this,
                "Round number transaction: {$this->formatted_amount}",
                'medium'
            );
        }
        
        // International transfer flag
        if ($this->is_international) {
            $complianceService->flagTransaction(
                $this,
                "International transfer flagged for review",
                'high'
            );
        }
        
        // Off-hours transaction flag
        $hour = now()->hour;
        if ($hour < 6 || $hour > 22) {
            $complianceService->flagTransaction(
                $this,
                "Transaction outside business hours",
                'low'
            );
        }
    }

    /**
     * Get validation rules.
     */
    public static function getValidationRules(): array
    {
        return [
            'account_id' => 'required|exists:accounts,id',
            'type' => 'required|in:deposit,withdrawal,transfer,payment,fee,interest,refund',
            'category' => 'required|string|max:100',
            'amount' => 'required|numeric|min:0.01',
            'currency' => 'required|string|size:3',
            'description' => 'required|string|max:500',
            'reference' => 'nullable|string|max:100',
            'beneficiary_id' => 'nullable|exists:beneficiaries,id',
            'notes' => 'nullable|string|max:1000'
        ];
    }

    /**
     * Activity log options.
     */
    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly([
                'transaction_id', 'type', 'category', 'amount', 'currency',
                'status', 'description', 'reference'
            ])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }

    /**
     * Audit configuration.
     */
    protected $auditInclude = [
        'transaction_id', 'type', 'category', 'amount', 'currency',
        'status', 'description', 'reference'
    ];

    protected $auditEvents = [
        'created', 'updated', 'deleted', 'restored'
    ];
}
