<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class ComplianceLog extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'account_id',
        'transaction_id',
        'kyc_document_id',
        'type',
        'category',
        'severity',
        'status',
        'description',
        'details',
        'admin_notes',
        'reviewed_by',
        'reviewed_at',
        'resolution_notes',
        'resolution_status',
        'risk_factors',
        'amount',
        'currency',
        'source_ip',
        'user_agent',
        'metadata',
    ];

    protected $casts = [
        'details' => 'array',
        'risk_factors' => 'array',
        'metadata' => 'array',
        'reviewed_at' => 'datetime',
        'amount' => 'decimal:2',
    ];

    // Relationships
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function account(): BelongsTo
    {
        return $this->belongsTo(Account::class);
    }

    public function transaction(): BelongsTo
    {
        return $this->belongsTo(Transaction::class);
    }

    public function kycDocument(): BelongsTo
    {
        return $this->belongsTo(KycDocument::class);
    }

    public function reviewer(): BelongsTo
    {
        return $this->belongsTo(User::class, 'reviewed_by');
    }

    // Helper methods
    public function getSeverityColorAttribute(): string
    {
        return match ($this->severity) {
            'critical' => 'bg-red-100 text-red-800',
            'high' => 'bg-orange-100 text-orange-800',
            'medium' => 'bg-yellow-100 text-yellow-800',
            'low' => 'bg-green-100 text-green-800',
            default => 'bg-gray-100 text-gray-800',
        };
    }

    public function getStatusColorAttribute(): string
    {
        return match ($this->status) {
            'pending' => 'bg-amber-100 text-amber-800',
            'under_review' => 'bg-blue-100 text-blue-800',
            'approved' => 'bg-green-100 text-green-800',
            'rejected' => 'bg-red-100 text-red-800',
            'escalated' => 'bg-purple-100 text-purple-800',
            default => 'bg-gray-100 text-gray-800',
        };
    }

    public function getTypeLabelAttribute(): string
    {
        return match ($this->type) {
            'transaction_flag' => 'Transaction Flag',
            'kyc_review' => 'KYC Review',
            'risk_assessment' => 'Risk Assessment',
            'compliance_check' => 'Compliance Check',
            default => ucfirst(str_replace('_', ' ', $this->type)),
        };
    }

    public function getCategoryLabelAttribute(): string
    {
        return match ($this->category) {
            'suspicious_activity' => 'Suspicious Activity',
            'high_value_transaction' => 'High Value Transaction',
            'unusual_pattern' => 'Unusual Pattern',
            'document_verification' => 'Document Verification',
            default => ucfirst(str_replace('_', ' ', $this->category)),
        };
    }

    public function isPending(): bool
    {
        return $this->status === 'pending';
    }

    public function isUnderReview(): bool
    {
        return $this->status === 'under_review';
    }

    public function isResolved(): bool
    {
        return in_array($this->status, ['approved', 'rejected']);
    }

    public function isHighPriority(): bool
    {
        return in_array($this->severity, ['high', 'critical']);
    }
}
