<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use OwenIt\Auditing\Contracts\Auditable;
use OwenIt\Auditing\Auditable as AuditableTrait;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;
use Akaunting\Money\Money;
use Akaunting\Money\Currency;

class Account extends Model implements Auditable
{
    use HasFactory, SoftDeletes, AuditableTrait, LogsActivity;

    /**
     * The attributes that are mass assignable.
     */
    protected $fillable = [
        'user_id',
        'account_number',
        'account_type',
        'account_name',
        'currency',
        'description',
        'balance',
        'available_balance',
        'hold_balance',
        'daily_transfer_limit',
        'monthly_transfer_limit',
        'daily_withdrawal_limit',
        'monthly_withdrawal_limit',
        'status',
        'verification_status',
        'activated_at',
        'closed_at',
        'closure_reason',
        'interest_rate',
        'interest_frequency',
        'monthly_maintenance_fee',
        'overdraft_limit',
        'overdraft_fee',
        'online_banking_enabled',
        'mobile_banking_enabled',
        'debit_card_enabled',
        'credit_card_enabled',
        'international_transfers_enabled',
        'investment_trading_enabled',
        'two_factor_required',
        'biometric_required',
        'security_preferences',
        'risk_level',
        'aml_monitoring_enabled',
        'last_risk_assessment',
        'parent_account_id',
        'is_master_account',
        'business_name',
        'business_type',
        'business_registration_number',
        'tax_identification_number',
        'investment_strategy',
        'risk_tolerance',
        'investment_preferences'
    ];

    /**
     * The attributes that should be cast.
     */
    protected $casts = [
        'balance' => 'decimal:4',
        'available_balance' => 'decimal:4',
        'hold_balance' => 'decimal:4',
        'daily_transfer_limit' => 'decimal:4',
        'monthly_transfer_limit' => 'decimal:4',
        'daily_withdrawal_limit' => 'decimal:4',
        'monthly_withdrawal_limit' => 'decimal:4',
        'interest_rate' => 'decimal:4',
        'monthly_maintenance_fee' => 'decimal:2',
        'overdraft_limit' => 'decimal:4',
        'overdraft_fee' => 'decimal:2',
        'activated_at' => 'datetime',
        'closed_at' => 'datetime',
        'last_risk_assessment' => 'datetime',
        'online_banking_enabled' => 'boolean',
        'mobile_banking_enabled' => 'boolean',
        'debit_card_enabled' => 'boolean',
        'credit_card_enabled' => 'boolean',
        'international_transfers_enabled' => 'boolean',
        'investment_trading_enabled' => 'boolean',
        'two_factor_required' => 'boolean',
        'biometric_required' => 'boolean',
        'aml_monitoring_enabled' => 'boolean',
        'is_master_account' => 'boolean',
        'security_preferences' => 'array',
        'investment_preferences' => 'array'
    ];

    /**
     * The attributes that should be hidden for serialization.
     */
    protected $hidden = [
        'user_id',
        'parent_account_id',
        'security_preferences',
        'business_registration_number',
        'tax_identification_number'
    ];

    /**
     * The attributes that should be appended.
     */
    protected $appends = [
        'formatted_balance',
        'formatted_available_balance',
        'account_status_label',
        'verification_status_label',
        'is_overdrawn',
        'is_active',
        'currency_symbol'
    ];

    /**
     * Get the user that owns the account.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the parent account.
     */
    public function parentAccount(): BelongsTo
    {
        return $this->belongsTo(Account::class, 'parent_account_id');
    }

    /**
     * Get the child accounts.
     */
    public function childAccounts(): HasMany
    {
        return $this->hasMany(Account::class, 'parent_account_id');
    }

    /**
     * Get the transactions for the account.
     */
    public function transactions(): HasMany
    {
        return $this->hasMany(Transaction::class, 'from_account_id');
    }

    /**
     * Get all transactions related to this account (both incoming and outgoing).
     */
    public function allTransactions()
    {
        return Transaction::where('from_account_id', $this->id)
            ->orWhere('to_account_id', $this->id);
    }


    /**
     * Get the formatted balance.
     */
    public function getFormattedBalanceAttribute(): string
    {
        return Money::{$this->currency}($this->balance)->format();
    }

    /**
     * Get the formatted available balance.
     */
    public function getFormattedAvailableBalanceAttribute(): string
    {
        return Money::{$this->currency}($this->available_balance)->format();
    }

    /**
     * Get the account status label.
     */
    public function getAccountStatusLabelAttribute(): string
    {
        return match($this->status) {
            'active' => 'Active',
            'inactive' => 'Inactive',
            'suspended' => 'Suspended',
            'frozen' => 'Frozen',
            'closed' => 'Closed',
            'pending' => 'Pending',
            default => 'Unknown'
        };
    }

    /**
     * Check if account is overdrawn.
     */
    public function getIsOverdrawnAttribute(): bool
    {
        return $this->balance < 0;
    }

    /**
     * Get verification status label.
     */
    public function getVerificationStatusLabelAttribute(): string
    {
        return match($this->verification_status) {
            'pending' => 'Pending Verification',
            'verified' => 'Verified',
            'rejected' => 'Rejected',
            default => 'Unknown'
        };
    }

    /**
     * Check if account is active.
     */
    public function getIsActiveAttribute(): bool
    {
        return $this->status === 'active';
    }

    /**
     * Get currency symbol.
     */
    public function getCurrencySymbolAttribute(): string
    {
        return \App\Helpers\CurrencyHelper::getSymbol($this->currency);
    }

    /**
     * Scope for active accounts.
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    /**
     * Scope for master accounts.
     */
    public function scopeMaster($query)
    {
        return $query->where('is_master_account', true);
    }

    /**
     * Scope for accounts by type.
     */
    public function scopeOfType($query, $type)
    {
        return $query->where('account_type', $type);
    }

    /**
     * Scope for accounts by currency.
     */
    public function scopeInCurrency($query, $currency)
    {
        return $query->where('currency', $currency);
    }


    /**
     * Check if account can perform transaction.
     */
    public function canPerformTransaction(float $amount, string $type = 'transfer'): bool
    {
        if ($this->status !== 'active') {
            return false;
        }

        if ($type === 'transfer') {
            $dailyLimit = $this->daily_transfer_limit ?? 0;
            $monthlyLimit = $this->monthly_transfer_limit ?? 0;
        } else {
            $dailyLimit = $this->daily_withdrawal_limit ?? 0;
            $monthlyLimit = $this->monthly_withdrawal_limit ?? 0;
        }

        // Check daily limit
        $todayTransactions = $this->transactions()
            ->where('type', $type)
            ->whereDate('created_at', today())
            ->sum('amount');

        if ($dailyLimit > 0 && ($todayTransactions + $amount) > $dailyLimit) {
            return false;
        }

        // Check monthly limit
        $monthlyTransactions = $this->transactions()
            ->where('type', $type)
            ->whereMonth('created_at', now()->month)
            ->whereYear('created_at', now()->year)
            ->sum('amount');

        if ($monthlyLimit > 0 && ($monthlyTransactions + $amount) > $monthlyLimit) {
            return false;
        }

        return true;
    }

    /**
     * Update account balance.
     */
    public function updateBalance(float $amount, string $type = 'credit'): bool
    {
        $oldBalance = $this->balance;
        
        if ($type === 'credit') {
            $this->balance += $amount;
        } else {
            $this->balance -= $amount;
        }

        $this->last_transaction_date = now();
        
        if ($this->save()) {
            // Log the balance change
            activity()
                ->performedOn($this)
                ->log("Account balance updated from {$oldBalance} to {$this->balance}");
            
            return true;
        }

        return false;
    }

    /**
     * Get validation rules.
     */
    public static function getValidationRules(): array
    {
        return [
            'account_type' => 'required|in:savings,checking,business,investment,fixed_deposit',
            'account_name' => 'required|string|max:255',
            'currency' => 'required|string|size:3',
            'description' => 'nullable|string|max:1000',
            'daily_transfer_limit' => 'nullable|numeric|min:0',
            'monthly_transfer_limit' => 'nullable|numeric|min:0',
            'daily_withdrawal_limit' => 'nullable|numeric|min:0',
            'monthly_withdrawal_limit' => 'nullable|numeric|min:0',
            'interest_rate' => 'nullable|numeric|min:0|max:1',
            'monthly_maintenance_fee' => 'nullable|numeric|min:0',
            'overdraft_limit' => 'nullable|numeric|min:0',
            'overdraft_fee' => 'nullable|numeric|min:0',
            'status' => 'required|in:active,inactive,suspended,closed,pending',
            'verification_status' => 'required|in:pending,verified,rejected',
            'risk_level' => 'required|in:low,medium,high',
            'risk_tolerance' => 'nullable|in:conservative,moderate,aggressive',
            'is_master_account' => 'boolean',
            'business_name' => 'nullable|string|max:255',
            'business_type' => 'nullable|string|max:255'
        ];
    }

    /**
     * Activity log options.
     */
    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly([
                'account_number', 'account_type', 'account_name', 'currency',
                'balance', 'status', 'verification_status', 'is_master_account'
            ])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }

    /**
     * Audit configuration.
     */
    protected $auditInclude = [
        'account_number', 'account_type', 'account_name', 'currency',
        'balance', 'status', 'verification_status', 'is_master_account'
    ];

    protected $auditEvents = [
        'created', 'updated', 'deleted', 'restored'
    ];
}
