<?php

namespace App\Livewire\User\Banking;

use App\Models\Transaction;
use App\Models\Beneficiary;
use App\Models\Account;
use Livewire\Component;
use Livewire\Attributes\Layout;
use Flux;

#[Layout('components.layouts.app')]
class TransferSuccess extends Component
{
    public $transactionId;
    public $transaction;
    public $beneficiary;
    public $account;

    public function mount($transaction)
    {
        $this->transactionId = $transaction;
        $this->loadData();
    }

    public function loadData()
    {
        $this->transaction = Transaction::find($this->transactionId);
        if (!$this->transaction) {
            return redirect()->route('user.dashboard');
        }

        // Load beneficiary if it exists
        if ($this->transaction->beneficiary_id) {
            $this->beneficiary = Beneficiary::find($this->transaction->beneficiary_id);
        }
        
        // Load account if it exists
        if ($this->transaction->from_account_id) {
            $this->account = Account::find($this->transaction->from_account_id);
        }
    }

    public function generateReceipt()
    {
        try {
            // Try different methods to load DOMPDF
            $pdf = null;
            
            // Method 1: Try using the service container
            try {
                $pdf = app('dompdf.wrapper');
            } catch (\Exception $e) {
                // Method 2: Try using the facade
                try {
                    $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadHTML('<h1>Test</h1>');
                } catch (\Exception $e2) {
                    // Method 3: Try using the class directly
                    $pdf = new \Barryvdh\DomPDF\PDF();
                }
            }
            
            $pdf->loadView('pdf.transfer-receipt', [
                'transaction' => $this->transaction,
                'beneficiary' => $this->beneficiary,
                'account' => $this->account,
            ])
            ->setPaper('a4', 'portrait')
            ->setOptions([
                'dpi' => 150,
                'defaultFont' => 'DejaVu Sans',
                'isRemoteEnabled' => true,
                'isHtml5ParserEnabled' => true,
            ]);

            $filename = 'transfer-receipt-' . $this->transaction->transaction_id . '.pdf';
            
            return response()->streamDownload(function () use ($pdf) {
                echo $pdf->output();
            }, $filename, [
                'Content-Type' => 'application/pdf',
                'Content-Disposition' => 'attachment; filename="' . $filename . '"'
            ]);

        } catch (\Exception $e) {
            Flux::toast('Error generating receipt: ' . $e->getMessage(), variant: 'danger');
            return null;
        }
    }

    public function render()
    {
        return view('livewire.user.banking.transfer-success');
    }
}
