<?php

namespace App\Livewire\User\Banking;

use Livewire\Component;
use Livewire\Attributes\Layout;
use Flux;
use App\Models\Code;
use App\Models\Transaction;
use App\Models\Account;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Mail;
use App\Mail\TwoFactorCodeMail;

#[Layout('components.layouts.app')]
class CodeVerification extends Component
{
    // Code inputs
    public $cotCode = '';
    public $taxCode = '';
    public $imfCode = '';
    public $twoFactorCode = '';

    // Status tracking
    public $cotStatus = false;
    public $taxStatus = false;
    public $imfStatus = false;
    public $twoFactorStatus = false;
    public $allCodesVerified = false;

    // Data
    public $pendingTransfer;
    public $account;
    public $beneficiary;
    public $codeRecord;
    public $twoFactorRequired = false;

    // Progress tracking
    public $currentStep = 1;
    public $totalSteps = 4;
    public $progress = 0;
    public $isLoading = false;
    public $availableSteps = [];
    public $currentStepIndex = 0;
    public $showNextStep = false;
    public $twoFactorCodeSent = false;
    public $twoFactorResendCooldown = 0;

    public function mount()
    {
        // Load session data first
        $this->loadData();
        
        // Initialize code verification
        $this->initializeCodeVerification();
    }

    public function loadData()
    {
        $this->pendingTransfer = session('pending_transfer');
        
        if (!$this->pendingTransfer) {
            return redirect()->route('dashboard');
        }

        // Load account data
        $this->account = Account::find($this->pendingTransfer['from_account_id']);
        
        // Load beneficiary data only if not a manual transfer
        if (!$this->pendingTransfer['metadata']['manual_transfer'] ?? false) {
            $this->beneficiary = \App\Models\Beneficiary::find($this->pendingTransfer['beneficiary_id']);
        }
    }

    public function initializeCodeVerification()
    {
        // Get or create code record for user
        $this->codeRecord = Code::where('user_id', auth()->id())->first();
        
        if (!$this->codeRecord) {
            // Create a new code record for testing
            $this->codeRecord = Code::create([
                'user_id' => auth()->id(),
                'cot_code' => strtoupper(Str::random(6)),
                'tax_code' => strtoupper(Str::random(6)),
                'imf_code' => strtoupper(Str::random(6)),
                'allow_local' => true,
                'allow_international' => true,
            ]);
        }

        // Check if 2FA is required based on account settings
        $this->twoFactorRequired = $this->account && $this->account->two_factor_required;
        
        // Initialize verification status as false (these track if codes have been verified)
        $this->cotStatus = false;
        $this->taxStatus = false;
        $this->imfStatus = false;
        
        // Build available steps array based on which codes are required
        $this->availableSteps = [];
        
        // Check if codes are required based on transfer type
        $codesRequired = false;
        if ($this->pendingTransfer['type'] === 'local' && $this->codeRecord->allow_local) {
            $codesRequired = true;
        } elseif ($this->pendingTransfer['type'] === 'international' && $this->codeRecord->allow_international) {
            $codesRequired = true;
        }
        
        if ($codesRequired) {
            if ($this->codeRecord->cot_status) {
                $this->availableSteps[] = [
                    'type' => 'cot',
                    'title' => 'COT Code Verification',
                    'description' => 'Enter your COT (Change of Terms) code to proceed.',
                    'icon' => 'key',
                    'code' => $this->codeRecord->cot_code,
                    'verified' => false
                ];
            }
            if ($this->codeRecord->tax_status) {
                $this->availableSteps[] = [
                    'type' => 'tax',
                    'title' => 'Tax Code Verification',
                    'description' => 'Enter your Tax code to proceed.',
                    'icon' => 'document-text',
                    'code' => $this->codeRecord->tax_code,
                    'verified' => false
                ];
            }
            if ($this->codeRecord->imf_status) {
                $this->availableSteps[] = [
                    'type' => 'imf',
                    'title' => 'IMF Code Verification',
                    'description' => 'Enter your IMF (International Monetary Fund) code to proceed.',
                    'icon' => 'globe-alt',
                    'code' => $this->codeRecord->imf_code,
                    'verified' => false
                ];
            }
        }
        
        if ($this->twoFactorRequired) {
            $this->availableSteps[] = [
                'type' => 'twofactor',
                'title' => 'Two-Factor Authentication',
                'description' => 'Enter the 6-digit code sent to your email.',
                'icon' => 'lock-closed',
                'code' => null, // No code display
                'verified' => false
            ];
        }
        
        $this->totalSteps = count($this->availableSteps);
        $this->currentStepIndex = 0;
        
        // Update progress
        $this->updateProgress();
    }

    public function updateProgress()
    {
        $verifiedCount = 0;
        
        // Count verified steps
        foreach ($this->availableSteps as $index => $step) {
            if ($step['verified']) {
                $verifiedCount++;
            }
        }
        
        $this->progress = $this->totalSteps > 0 ? ($verifiedCount / $this->totalSteps) * 100 : 0;
        $this->allCodesVerified = $verifiedCount === $this->totalSteps && $this->totalSteps > 0;
        
        // Update current step index to next unverified step
        $this->currentStepIndex = $verifiedCount;
        $this->currentStep = $verifiedCount + 1;
    }

    public function submitCode()
    {
        $this->isLoading = true;
        
        // Get current step
        if ($this->currentStepIndex >= count($this->availableSteps)) {
            $this->isLoading = false;
            return;
        }
        
        $currentStep = $this->availableSteps[$this->currentStepIndex];
        $isValid = false;
        
        // Validate based on step type
        switch ($currentStep['type']) {
            case 'cot':
                $isValid = $this->codeRecord->cot_code && $this->codeRecord->cot_code === $this->cotCode;
                if ($isValid) {
                    $this->cotStatus = true;
                    $this->cotCode = '';
                }
                break;
            case 'tax':
                $isValid = $this->codeRecord->tax_code && $this->codeRecord->tax_code === $this->taxCode;
                if ($isValid) {
                    $this->taxStatus = true;
                    $this->taxCode = '';
                }
                break;
            case 'imf':
                $isValid = $this->codeRecord->imf_code && $this->codeRecord->imf_code === $this->imfCode;
                if ($isValid) {
                    $this->imfStatus = true;
                    $this->imfCode = '';
                }
                break;
            case 'twofactor':
                $sessionCode = session('two_factor_code');
                $expires = session('two_factor_expires');
                
                // Check if code exists and hasn't expired
                if ($sessionCode && $expires && now()->timestamp < $expires) {
                    $isValid = $sessionCode === $this->twoFactorCode;
                    if ($isValid) {
                        $this->twoFactorStatus = true;
                        $this->twoFactorCode = '';
                        // Clear the session code after successful verification
                        session()->forget(['two_factor_code', 'two_factor_expires']);
                    }
                } else {
                    $isValid = false;
                    Flux::toast('2FA code has expired. Please request a new one.', variant: 'warning');
                }
                break;
        }
        
        if ($isValid) {
            // Mark step as verified
            $this->availableSteps[$this->currentStepIndex]['verified'] = true;
            $this->updateProgress();
            
            // Keep loading state and show next step after delay
            $this->showNextStep = true;
            
            // Use JavaScript to delay showing next step
            $this->dispatch('show-next-step');
        } else {
            Flux::toast('Invalid code. Please try again.', variant: 'danger');
            $this->isLoading = false;
        }
    }

    public function proceedToNextStep()
    {
        $this->isLoading = false;
        $this->showNextStep = false;
    }

    public function sendTwoFactorCode()
    {
        if ($this->twoFactorResendCooldown > 0) {
            Flux::toast('Please wait before requesting another code.', variant: 'warning');
            return;
        }

        try {
            // Generate 6-digit code
            $code = str_pad(random_int(0, 999999), 6, '0', STR_PAD_LEFT);
            
            // Store code in session with expiration (5 minutes)
            session([
                'two_factor_code' => $code,
                'two_factor_expires' => now()->addMinutes(5)->timestamp
            ]);

            // Send email using Laravel mail class
            $user = auth()->user();
            \Mail::to($user->email)->send(new TwoFactorCodeMail($code, $user, $this->pendingTransfer));

            $this->twoFactorCodeSent = true;
            $this->twoFactorResendCooldown = 60; // 60 seconds cooldown
            
            Flux::toast('2FA code sent to your email.', variant: 'success');
            
            // Start countdown
            $this->dispatch('start-resend-countdown');

        } catch (\Exception $e) {
            Flux::toast('Failed to send 2FA code. Please try again.', variant: 'danger');
            \Log::error('2FA code send error: ' . $e->getMessage());
        }
    }

    public function resendTwoFactorCode()
    {
        $this->sendTwoFactorCode();
    }

    public function completeTransfer()
    {
        if (!$this->allCodesVerified) {
            Flux::toast('Please verify all required codes first.', variant: 'danger');
            return;
        }

        try {
            // Create transaction
            $transaction = Transaction::create([
                'transaction_id' => 'TXN-' . strtoupper(uniqid()),
                'from_account_id' => $this->pendingTransfer['from_account_id'],
                'to_account_id' => $this->pendingTransfer['to_account_id'],
                'beneficiary_id' => $this->pendingTransfer['beneficiary_id'],
                'amount' => $this->pendingTransfer['amount'],
                'currency' => $this->pendingTransfer['currency'],
                'exchange_rate' => null,
                'fee_amount' => $this->pendingTransfer['fee_amount'],
                'net_amount' => $this->pendingTransfer['net_amount'],
                'type' => 'transfer',
                'category' => 'banking',
                'subcategory' => $this->pendingTransfer['type'],
                'status' => 'pending',
                'description' => $this->pendingTransfer['description'],
                'notes' => $this->pendingTransfer['notes'] ?? '',
                'transfer_method' => $this->pendingTransfer['transfer_method'],
                'scheduled_at' => $this->pendingTransfer['scheduled_at'],
                'user_id' => auth()->id(),
                'metadata' => $this->pendingTransfer['metadata'] ?? [],
            ]);

            // Update account balance
            $account = Account::find($this->pendingTransfer['from_account_id']);
            $account->increment('hold_balance', $this->pendingTransfer['net_amount']);
            $account->decrement('available_balance', $this->pendingTransfer['net_amount']);
            $account->decrement('balance', $this->pendingTransfer['net_amount']);

            // Complete the transaction
            $transaction->complete();

            // Send notification
            $notificationService = app(\App\Services\NotificationService::class);
            $notificationService->notifyTransferCompleted(
                auth()->user(),
                $this->pendingTransfer['amount'],
                $this->pendingTransfer['currency'],
                $this->pendingTransfer['beneficiary_name'] ?? 'Unknown',
                $transaction->transaction_id
            );

            // Clear session data
            session()->forget('pending_transfer');

            Flux::toast('Transfer completed successfully!', variant: 'success');
            
            // Redirect to success page with transaction ID
            return redirect()->route('user.transfer.success', ['transaction' => $transaction->id]);

        } catch (\Exception $e) {
            Flux::toast('Transfer failed: ' . $e->getMessage(), variant: 'danger');
            \Log::error('Transfer completion error: ' . $e->getMessage());
        }
    }

    public function render()
    {
        return view('livewire.user.banking.code-verification');
    }
}