<?php

namespace App\Livewire\User\Banking;

use App\Models\Beneficiary;
use App\Models\Country;
use Flux;
use Livewire\Component;
use Livewire\WithFileUploads;
use Livewire\Attributes\Layout;

#[Layout('components.layouts.app')]
class BeneficiaryManagement extends Component
{
    use WithFileUploads;

    // Form inputs
    public $name = '';
    public $type = 'domestic';
    public $email = '';
    public $phone = '';
    public $bank_name = '';
    public $account_number = '';
    public $iban = '';
    public $swift_code = '';
    public $address = '';
    public $country_id = '';
    public $currency = 'USD';

    // UI state
    public $showForm = false;
    public $editingBeneficiary = null;

    protected $rules = [
        'name' => 'required|string|max:255',
        'type' => 'required|in:domestic,international',
        'email' => 'nullable|email|max:255',
        'phone' => 'nullable|string|max:255',
        'bank_name' => 'required|string|max:255',
        'account_number' => 'required|string|max:255',
        'iban' => 'nullable|string|max:34',
        'swift_code' => 'nullable|string|max:11',
        'address' => 'nullable|string|max:500',
        'country_id' => 'required|exists:countries,id',
        'currency' => 'required|string|size:3',
    ];

    protected $messages = [
        'name.required' => 'Beneficiary name is required.',
        'type.required' => 'Beneficiary type is required.',
        'type.in' => 'Invalid beneficiary type selected.',
        'bank_name.required' => 'Bank name is required.',
        'account_number.required' => 'Account number is required.',
        'country_id.required' => 'Country is required.',
        'country_id.exists' => 'Selected country is invalid.',
        'currency.required' => 'Currency is required.',
        'currency.size' => 'Currency must be exactly 3 characters.',
    ];

    public function mount()
    {
        // Initialize component
    }

    public function updatedCountryId()
    {
        // Auto-select currency when country changes
        if ($this->country_id) {
            $country = Country::find($this->country_id);
            if ($country && $country->currency) {
                $this->currency = $country->currency;
            }
        }
    }

    public function showAddForm()
    {
        $this->resetForm();
        $this->showForm = true;
    }

    public function showEditForm($beneficiaryId)
    {
        $beneficiary = Beneficiary::where('id', $beneficiaryId)
            ->where('user_id', auth()->id())
            ->first();

        if ($beneficiary) {
            $this->editingBeneficiary = $beneficiary;
            $this->name = $beneficiary->name;
            $this->type = $beneficiary->type;
            $this->email = $beneficiary->email;
            $this->phone = $beneficiary->phone;
            $this->bank_name = $beneficiary->bank_name;
            $this->account_number = $beneficiary->account_number;
            $this->country_id = $beneficiary->country_id ?? '';
            $this->currency = $beneficiary->currency;
            $this->showForm = true;
        }
    }

    public function hideForm()
    {
        $this->showForm = false;
        $this->editingBeneficiary = null;
        $this->resetForm();
    }

    public function saveBeneficiary()
    {
        $this->validate();

        try {
            // Get country information
            $country = Country::find($this->country_id);
            
            $data = [
                'user_id' => auth()->id(),
                'beneficiary_id' => 'BEN-' . strtoupper(uniqid()),
                'name' => $this->name,
                'type' => $this->type,
                'email' => $this->email,
                'phone' => $this->phone,
                'bank_name' => $this->bank_name,
                'account_number' => $this->account_number,
                'iban' => $this->iban,
                'swift_code' => $this->swift_code,
                'address' => $this->address,
                'country' => $country ? $country->name : '',
                'currency' => $this->currency,
                'verification_status' => 'verified',
                'verification_level' => 'basic',
                'risk_level' => 'medium',
                'active' => true,
            ];

            if ($this->editingBeneficiary) {
                $this->editingBeneficiary->update($data);
                $message = 'Beneficiary updated successfully!';
            } else {
                Beneficiary::create($data);
                $message = 'Beneficiary added successfully!';
                
                // Send notification for new beneficiary
                $notificationService = app(\App\Services\NotificationService::class);
                $notificationService->notifyBeneficiaryAdded(
                    auth()->user(),
                    $this->name,
                    $this->account_number
                );
            }

            Flux::toast($message, variant: 'success');
            $this->hideForm();

        } catch (\Exception $e) {
            Flux::toast('Failed to save beneficiary: ' . $e->getMessage(), variant: 'danger');
        }
    }

    public function deleteBeneficiary($beneficiaryId)
    {
        try {
            $beneficiary = Beneficiary::where('id', $beneficiaryId)
                ->where('user_id', auth()->id())
                ->first();

            if ($beneficiary) {
                $beneficiary->delete();
                Flux::toast('Beneficiary deleted successfully!', variant: 'success');
            }
        } catch (\Exception $e) {
            Flux::toast('Failed to delete beneficiary: ' . $e->getMessage(), variant: 'danger');
        }
    }

    public function updatedType()
    {
        // Clear international-specific fields when switching to domestic
        if ($this->type === 'domestic') {
            $this->iban = '';
            $this->swift_code = '';
            $this->address = '';
        }
    }

    public function resetForm()
    {
        $this->name = '';
        $this->type = 'domestic';
        $this->email = '';
        $this->phone = '';
        $this->bank_name = '';
        $this->account_number = '';
        $this->iban = '';
        $this->swift_code = '';
        $this->address = '';
        $this->country_id = '';
        $this->currency = 'USD';
    }

    public function getBeneficiariesProperty()
    {
        return Beneficiary::where('user_id', auth()->id())
            ->orderBy('created_at', 'desc')
            ->get();
    }

    public function getCountriesProperty()
    {
        return Country::orderBy('name')->get();
    }

    public function getCurrenciesProperty()
    {
        return Country::getAllCurrencies();
    }

    public function render()
    {
        return view('livewire.user.banking.beneficiary-management');
    }
}
