<?php

namespace App\Livewire\User;

use App\Models\Account;
use App\Models\Transaction;
use App\Helpers\CurrencyHelper;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;
use Livewire\Attributes\Layout;
use Livewire\WithPagination;

#[Layout('components.layouts.app')]
class AccountHistory extends Component
{
    use WithPagination;

    public $masterAccount;
    public $currentBalance;
    public $availableBalance;
    public $currency;
    
    // Filter properties
    public $dateFrom = '';
    public $dateTo = '';
    public $transactionType = '';
    public $status = '';
    public $amountFrom = '';
    public $amountTo = '';
    
    // Pagination
    public $perPage = 10;

    protected $queryString = [
        'dateFrom' => ['except' => ''],
        'dateTo' => ['except' => ''],
        'transactionType' => ['except' => ''],
        'status' => ['except' => ''],
        'amountFrom' => ['except' => ''],
        'amountTo' => ['except' => ''],
    ];

    public function mount()
    {
        $this->loadMasterAccount();
    }

    public function loadMasterAccount()
    {
        $user = Auth::user();
        $this->masterAccount = $user->accounts()->master()->first();
        
        if ($this->masterAccount) {
            $this->currentBalance = $this->masterAccount->balance;
            $this->availableBalance = $this->masterAccount->available_balance;
            $this->currency = $this->masterAccount->currency;
        }
    }

    public function getTransactionsProperty()
    {
        if (!$this->masterAccount) {
            return new \Illuminate\Pagination\LengthAwarePaginator([], 0, $this->perPage);
        }

        $query = Transaction::where(function ($q) {
            $q->where('from_account_id', $this->masterAccount->id)
              ->orWhere('to_account_id', $this->masterAccount->id);
        });

        // Apply filters
        if ($this->dateFrom) {
            $query->whereDate('created_at', '>=', $this->dateFrom);
        }

        if ($this->dateTo) {
            $query->whereDate('created_at', '<=', $this->dateTo);
        }

        if ($this->transactionType) {
            $query->where('type', $this->transactionType);
        }

        if ($this->status) {
            $query->where('status', $this->status);
        }

        if ($this->amountFrom) {
            $query->where('amount', '>=', $this->amountFrom);
        }

        if ($this->amountTo) {
            $query->where('amount', '<=', $this->amountTo);
        }

        return $query->orderBy('created_at', 'desc')->paginate($this->perPage);
    }

    public function clearFilters()
    {
        $this->dateFrom = '';
        $this->dateTo = '';
        $this->transactionType = '';
        $this->status = '';
        $this->amountFrom = '';
        $this->amountTo = '';
        $this->resetPage();
    }

    public function getTransactionTypeLabel($type)
    {
        return match ($type) {
            'deposit' => 'Deposit',
            'withdrawal' => 'Withdrawal',
            'transfer' => 'Transfer',
            'payment' => 'Payment',
            'fee' => 'Fee',
            'interest' => 'Interest',
            'refund' => 'Refund',
            default => ucfirst($type),
        };
    }

    public function getTransactionTypeIcon($type)
    {
        return match ($type) {
            'deposit' => 'arrow-down-circle',
            'withdrawal' => 'arrow-up-circle',
            'transfer' => 'arrow-right-left',
            'payment' => 'credit-card',
            'fee' => 'minus-circle',
            'interest' => 'plus-circle',
            'refund' => 'arrow-uturn-left',
            default => 'currency-dollar',
        };
    }

    public function getTransactionTypeColor($type)
    {
        return match ($type) {
            'deposit' => 'text-green-600',
            'withdrawal' => 'text-red-600',
            'transfer' => 'text-blue-600',
            'payment' => 'text-purple-600',
            'fee' => 'text-orange-600',
            'interest' => 'text-green-600',
            'refund' => 'text-blue-600',
            default => 'text-gray-600',
        };
    }

    public function getStatusLabel($status)
    {
        return match ($status) {
            'completed' => 'Completed',
            'pending' => 'Pending',
            'failed' => 'Failed',
            'cancelled' => 'Cancelled',
            'processing' => 'Processing',
            default => ucfirst($status),
        };
    }

    public function getStatusColor($status)
    {
        return match ($status) {
            'completed' => 'bg-green-100 text-green-800',
            'pending' => 'bg-amber-100 text-amber-800',
            'failed' => 'bg-red-100 text-red-800',
            'cancelled' => 'bg-gray-100 text-gray-800',
            'processing' => 'bg-blue-100 text-blue-800',
            default => 'bg-gray-100 text-gray-800',
        };
    }

    public function getStatusIcon($status)
    {
        return match ($status) {
            'completed' => 'check-circle',
            'pending' => 'clock',
            'failed' => 'x-circle',
            'cancelled' => 'minus-circle',
            'processing' => 'arrow-path',
            default => 'question-mark-circle',
        };
    }

    public function getTransactionAmount($transaction)
    {
        $currency = $transaction->currency ?? $this->currency ?? 'USD';
        $amount = (float)$transaction->amount;
        
        if ($transaction->to_account_id === $this->masterAccount->id) {
            // Money coming in
            return '+' . CurrencyHelper::format($amount, $currency);
        } else {
            // Money going out
            return '-' . CurrencyHelper::format($amount, $currency);
        }
    }

    public function getTransactionAmountColor($transaction)
    {
        if ($transaction->to_account_id === $this->masterAccount->id) {
            return 'text-green-600';
        } else {
            return 'text-red-600';
        }
    }

    public function getCurrencySymbol($currency = null)
    {
        $currency = $currency ?? $this->currency ?? 'USD';
        return CurrencyHelper::getSymbol($currency);
    }

    public function render()
    {
        $transactions = $this->getTransactionsProperty();
        
        return view('livewire.user.account-history', [
            'transactions' => $transactions,
        ]);
    }
}
