<?php

namespace App\Livewire\Layout;

use App\Models\Notification;
use App\Services\NotificationService;
use Livewire\Component;

class NotificationDropdown extends Component
{
    public $notifications = [];
    public $unreadCount = 0;

    protected $listeners = ['notificationAdded' => 'loadNotifications'];

    public function mount()
    {
        $this->loadNotifications();
    }

    public function loadNotifications()
    {
        if (auth()->check()) {
            $notificationService = app(NotificationService::class);
            $this->notifications = $notificationService->getRecentNotifications(auth()->user(), 10);
            $this->unreadCount = $notificationService->getUnreadCount(auth()->user());
        }
    }

    public function markAsRead($notificationId)
    {
        $notification = Notification::find($notificationId);
        if ($notification && $notification->user_id === auth()->id()) {
            $notification->markAsRead();
            $this->loadNotifications();
        }
    }

    public function markAllAsRead()
    {
        if (auth()->check()) {
            $notificationService = app(NotificationService::class);
            $notificationService->markAllAsRead(auth()->user());
            $this->loadNotifications();
        }
    }

    public function deleteNotification($notificationId)
    {
        $notification = Notification::find($notificationId);
        if ($notification && $notification->user_id === auth()->id()) {
            $notification->delete();
            $this->loadNotifications();
        }
    }

    public function deleteAllNotifications()
    {
        if (auth()->check()) {
            Notification::where('user_id', auth()->id())->delete();
            $this->loadNotifications();
        }
    }

    public function getNotificationIcon($type)
    {
        return match($type) {
            'login_ip_change' => 'shield-exclamation',
            'new_user_welcome' => 'hand-wave',
            'kyc_approved' => 'check-circle',
            'kyc_rejected' => 'x-circle',
            'virtual_card_created' => 'credit-card',
            'transfer_completed' => 'arrow-right-left',
            'inter_account_transfer' => 'arrow-right-left',
            'cheque_deposit' => 'document-text',
            'password_changed' => 'key',
            'pin_changed' => 'lock-closed',
            'beneficiary_added' => 'user-plus',
            'two_factor_enabled' => 'shield-check',
            'two_factor_disabled' => 'shield-exclamation',
            'account_created' => 'plus-circle',
            default => 'bell',
        };
    }

    public function getNotificationColor($type)
    {
        return match($type) {
            'login_ip_change' => 'bg-red-500',
            'new_user_welcome' => 'bg-blue-500',
            'kyc_approved' => 'bg-green-500',
            'kyc_rejected' => 'bg-red-500',
            'virtual_card_created' => 'bg-blue-500',
            'transfer_completed' => 'bg-green-500',
            'inter_account_transfer' => 'bg-blue-500',
            'cheque_deposit' => 'bg-yellow-500',
            'password_changed' => 'bg-orange-500',
            'pin_changed' => 'bg-orange-500',
            'beneficiary_added' => 'bg-purple-500',
            'two_factor_enabled' => 'bg-green-500',
            'two_factor_disabled' => 'bg-red-500',
            'account_created' => 'bg-green-500',
            default => 'bg-gray-500',
        };
    }

    public function render()
    {
        return view('livewire.layout.notification-dropdown');
    }
}
