<?php

namespace App\Livewire\Auth;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Livewire\Component;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Validate;
use App\Mail\EmailVerificationCodeMail;
use Flux;

#[Layout('components.layouts.auth')]
class VerifyEmail extends Component
{
    #[Validate('required|string|size:6')]
    public string $verificationCode = '';

    public bool $codeSent = false;
    public bool $isLoading = false;
    public int $resendCooldown = 0;
    public int $attempts = 0;
    public int $maxAttempts = 3;

    public function mount(Request $request)
    {
        if ($request->user()->hasVerifiedEmail()) {
            $this->redirect(route('dashboard'), navigate: true);
        }

        $this->attempts = $request->user()->email_verification_attempts ?? 0;
        
        // Check if there's already a valid verification code
        if ($request->user()->email_verification_code && !$request->user()->isEmailVerificationCodeExpired()) {
            $this->codeSent = true;
        }
    }

    public function sendVerificationCode()
    {
        if ($this->resendCooldown > 0) {
            Flux::toast('Please wait before requesting another code.', variant: 'warning');
            return;
        }

        $user = auth()->user();
        
        if ($user->hasVerifiedEmail()) {
            $this->redirect(route('dashboard'), navigate: true);
            return;
        }

        try {
            $code = $user->generateEmailVerificationCode();
            
            // Send email with code
            Mail::to($user->email)->send(new EmailVerificationCodeMail($code, $user));

            $this->codeSent = true;
            $this->resendCooldown = 60; // 60 seconds cooldown
            
            Flux::toast('Verification code sent to your email.', variant: 'success');
            
            // Start countdown
            $this->dispatch('start-resend-countdown');

        } catch (\Exception $e) {
            Flux::toast('Failed to send verification code. Please try again.', variant: 'danger');
            \Log::error('Email verification code send error: ' . $e->getMessage());
        }
    }

    public function verifyCode()
    {
        $this->validate();

        $user = auth()->user();
        
        if ($user->hasVerifiedEmail()) {
            $this->redirect(route('dashboard'), navigate: true);
            return;
        }

        if ($this->attempts >= $this->maxAttempts) {
            Flux::toast('Too many failed attempts. Please request a new code.', variant: 'danger');
            return;
        }

        $this->isLoading = true;

        if ($user->verifyEmailCode($this->verificationCode)) {
            Flux::toast('Email verified successfully!', variant: 'success');
            $this->redirect(route('dashboard'), navigate: true);
        } else {
            $this->attempts = $user->fresh()->email_verification_attempts;
            $this->verificationCode = '';
            
            if ($this->attempts >= $this->maxAttempts) {
                Flux::toast('Too many failed attempts. Please request a new code.', variant: 'danger');
            } else {
                $remaining = $this->maxAttempts - $this->attempts;
                Flux::toast("Invalid code. {$remaining} attempts remaining.", variant: 'danger');
            }
        }

        $this->isLoading = false;
    }

    public function resendCode()
    {
        $this->sendVerificationCode();
    }

    public function logout()
    {
        Auth::logout();
        $this->redirect(route('login'), navigate: true);
    }

    public function render()
    {
        return view('livewire.auth.verify-email');
    }
}
