<?php

namespace App\Livewire\Auth;

use App\Models\Country;
use App\Models\User;
use App\Models\UserProfile;
use App\Models\Account;
use App\Helpers\SettingsHelper;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;
use Livewire\Component;
use Livewire\Attributes\Layout;
use Spatie\Honeypot\Http\Livewire\Concerns\UsesSpamProtection;
use Spatie\Honeypot\Http\Livewire\Concerns\HoneypotData;

#[Layout('components.layouts.auth')]
class Register extends Component
{
    use UsesSpamProtection;

    public string $name = '';
    public string $username = '';
    public string $email = '';
    public string $password = '';
    public string $password_confirmation = '';
    public string $pin = '';
    public string $pin_confirmation = '';
    public string $customer_type = 'individual';
    
    public HoneypotData $extraFields;
    

    protected $rules = [
        'name' => 'required|string|max:255',
        'username' => 'required|string|max:255|unique:users',
        'email' => 'required|string|email|max:255|unique:users',
        'password' => 'required|string|min:8|confirmed',
        'pin' => 'required|string|size:6|regex:/^[0-9]{6}$/|confirmed',
        'customer_type' => 'required|in:individual,business,corporate',
    ];

    public function mount()
    {
        $this->extraFields = new HoneypotData();
    }

    public function createAccount()
    {
        $this->protectAgainstSpam();
        
        try {
            $this->validate();
        } catch (ValidationException $e) {
            // Validation failed, errors will be displayed automatically
            return;
        }

        try {
            DB::beginTransaction();

            // Check if email verification is enabled
            $isEmailVerificationEnabled = SettingsHelper::isEmailVerificationEnabled();
            
            $user = User::create([
                'name' => $this->name,
                'username' => $this->username,
                'email' => $this->email,
                'password' => Hash::make($this->password),
                'pin' => $this->pin,
                'customer_type' => $this->customer_type,
                'customer_status' => $isEmailVerificationEnabled ? 'pending_verification' : 'active',
            ]);

            if (!$isEmailVerificationEnabled) {
                $user->markEmailAsVerified();
            }

            // Assign customer role
            $user->assignRole('customer');

            Auth::login($user);
            
            // Send welcome notification
            $notificationService = app(\App\Services\NotificationService::class);
            $notificationService->notifyNewUserWelcome($user);
            
            DB::commit();
            
            // Redirect based on email verification requirement
            if ($isEmailVerificationEnabled) {
                // Automatically send verification code after registration
                try {
                    $code = $user->generateEmailVerificationCode();
                    \Mail::to($user->email)->send(new \App\Mail\EmailVerificationCodeMail($code, $user));
                } catch (\Exception $e) {
                    \Log::error('Failed to send verification code after registration: ' . $e->getMessage());
                }
                
                $this->redirect(route('verification.notice'), navigate: true);
            } else {
                $this->redirect(route('profile.complete'), navigate: true);
            }

        } catch (\Exception $e) {
            DB::rollBack();
            
            // Log the error for debugging
            \Log::error('Registration failed: ' . $e->getMessage(), [
                'user_data' => [
                    'name' => $this->name,
                    'username' => $this->username,
                    'email' => $this->email,
                    'customer_type' => $this->customer_type,
                ],
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            $this->addError('general', 'Registration failed: ' . $e->getMessage());
        }
    }


    public function render()
    {
        return view('livewire.auth.register');
    }
}
