<?php

namespace App\Livewire\Auth;

use App\Models\UserProfile;
use App\Models\Country;
use App\Models\Account;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Validate;

#[Layout('components.layouts.app')]
class ProfileCompletion extends Component
{
    public int $currentStep = 1;
    public int $totalSteps = 4;
    public bool $profileCompleted = false;

    // Step 1: Personal Information
    #[Validate('required|string|max:100')]
    public string $first_name = '';
    
    #[Validate('required|string|max:100')]
    public string $last_name = '';
    
    #[Validate('nullable|string|max:100')]
    public string $middle_name = '';
    
    #[Validate('required|date|before:today')]
    public string $date_of_birth = '';
    
    #[Validate('required|in:male,female,other,prefer_not_to_say')]
    public string $gender = '';
    
    #[Validate('nullable|in:single,married,divorced,widowed,separated')]
    public string $marital_status = '';

    // Step 2: Contact Information
    #[Validate('required|string|max:255')]
    public string $address_line_1 = '';
    
    #[Validate('required|string|max:100')]
    public string $city = '';
    
    #[Validate('nullable|string|max:100')]
    public string $state_province = '';
    
    #[Validate('required|string|max:20')]
    public string $postal_code = '';
    
    #[Validate('required|string|max:100')]
    public string $country = '';
    
    #[Validate('required|string|max:20')]
    public string $phone_number = '';

    // Step 3: Employment Information
    #[Validate('required|string|max:255')]
    public string $occupation = '';
    
    #[Validate('nullable|string|max:255')]
    public string $employer = '';
    
    #[Validate('nullable|numeric|min:0')]
    public ?float $annual_income = null;
    
    #[Validate('nullable|string|max:255')]
    public string $income_source = '';

    // Step 4: Emergency Contact
    #[Validate('nullable|string|max:255')]
    public string $emergency_contact_name = '';
    
    #[Validate('nullable|string|max:20')]
    public string $emergency_contact_phone = '';
    
    #[Validate('nullable|string|max:100')]
    public string $emergency_contact_relationship = '';

    public array $countries = [];

    public function mount()
    {
        $user = Auth::user();
        $profile = $user->profile;

        // Load countries for dropdown
        $this->countries = Country::orderBy('name')->pluck('name', 'name')->toArray();

        // Pre-fill if profile exists
        if ($profile) {
            $this->first_name = $profile->first_name ?? '';
            $this->last_name = $profile->last_name ?? '';
            $this->middle_name = $profile->middle_name ?? '';
            $this->date_of_birth = $profile->date_of_birth ? $profile->date_of_birth->format('Y-m-d') : '';
            $this->gender = $profile->gender ?? '';
            $this->marital_status = $profile->marital_status ?? '';
            $this->address_line_1 = $profile->address_line_1 ?? '';
            $this->city = $profile->city ?? '';
            $this->state_province = $profile->state_province ?? '';
            $this->postal_code = $profile->postal_code ?? '';
            $this->country = $profile->country ?? '';
            $this->phone_number = $profile->phone_number ?? '';
            $this->occupation = $profile->occupation ?? '';
            $this->annual_income = $profile->annual_income ?? null;
            $this->emergency_contact_name = $profile->emergency_contact_name ?? '';
            $this->emergency_contact_phone = $profile->emergency_contact_phone ?? '';
            $this->emergency_contact_relationship = $profile->emergency_contact_relationship ?? '';
        } else {
            // Pre-fill from user registration data if available
            if ($user->name) {
                $nameParts = explode(' ', $user->name, 2);
                $this->first_name = $nameParts[0] ?? '';
                $this->last_name = $nameParts[1] ?? '';
            }
        }
    }

    public function nextStep()
    {
        $this->validateCurrentStep();
        
        if ($this->currentStep < $this->totalSteps) {
            $this->currentStep++;
        }
    }

    public function previousStep()
    {
        if ($this->currentStep > 1) {
            $this->currentStep--;
        }
    }

    public function goToStep(int $step)
    {
        if ($step >= 1 && $step <= $this->totalSteps) {
            $this->currentStep = $step;
        }
    }

    protected function validateCurrentStep()
    {
        $rules = [];
        
        switch ($this->currentStep) {
            case 1:
                $rules = [
                    'first_name' => 'required|string|max:100',
                    'last_name' => 'required|string|max:100',
                    'middle_name' => 'nullable|string|max:100',
                    'date_of_birth' => 'required|date|before:today',
                    'gender' => 'required|in:male,female,other,prefer_not_to_say',
                    'marital_status' => 'nullable|in:single,married,divorced,widowed,separated',
                ];
                break;
            case 2:
                $rules = [
                    'address_line_1' => 'required|string|max:255',
                    'city' => 'required|string|max:100',
                    'state_province' => 'nullable|string|max:100',
                    'postal_code' => 'required|string|max:20',
                    'country' => 'required|string|max:100',
                    'phone_number' => 'required|string|max:20',
                ];
                break;
            case 3:
                $rules = [
                    'occupation' => 'required|string|max:255',
                    'annual_income' => 'nullable|numeric|min:0',
                ];
                break;
            case 4:
                $rules = [
                    'emergency_contact_name' => 'nullable|string|max:255',
                    'emergency_contact_phone' => 'nullable|string|max:20',
                    'emergency_contact_relationship' => 'nullable|string|max:100',
                ];
                break;
        }

        $this->validate($rules);
    }

    public function completeProfile()
    {
        // Validate all steps
        $this->validate([
            'first_name' => 'required|string|max:100',
            'last_name' => 'required|string|max:100',
            'middle_name' => 'nullable|string|max:100',
            'date_of_birth' => 'required|date|before:today',
            'gender' => 'required|in:male,female,other,prefer_not_to_say',
            'marital_status' => 'nullable|in:single,married,divorced,widowed,separated',
            'address_line_1' => 'required|string|max:255',
            'city' => 'required|string|max:100',
            'state_province' => 'nullable|string|max:100',
            'postal_code' => 'required|string|max:20',
            'country' => 'required|string|max:100',
            'phone_number' => 'required|string|max:20',
            'occupation' => 'required|string|max:255',
            'annual_income' => 'nullable|numeric|min:0',
            'emergency_contact_name' => 'nullable|string|max:255',
            'emergency_contact_phone' => 'nullable|string|max:20',
            'emergency_contact_relationship' => 'nullable|string|max:100',
        ]);

        $user = Auth::user();
        
        // Create or update user profile
        $profile = UserProfile::updateOrCreate(
            ['user_id' => $user->id],
            [
                'first_name' => $this->first_name,
                'last_name' => $this->last_name,
                'middle_name' => $this->middle_name,
                'date_of_birth' => $this->date_of_birth,
                'gender' => $this->gender,
                'marital_status' => $this->marital_status,
                'address_line_1' => $this->address_line_1,
                'city' => $this->city,
                'state_province' => $this->state_province,
                'postal_code' => $this->postal_code,
                'country' => $this->country,
                'phone_number' => $this->phone_number,
                'occupation' => $this->occupation,
                'annual_income' => $this->annual_income,
                'emergency_contact_name' => $this->emergency_contact_name,
                'emergency_contact_phone' => $this->emergency_contact_phone,
                'emergency_contact_relationship' => $this->emergency_contact_relationship,
            ]
        );

        $this->profileCompleted = true;
    }

    public function proceedToAccountCreation()
    {
        $user = Auth::user();
        
        // Check if user already has an account
        if ($user->accounts()->exists()) {
            return $this->redirect(route('dashboard'), navigate: true);
        }
        
        return $this->redirect(route('account.create'), navigate: true);
    }

    public function getCompletionPercentageProperty(): int
    {
        $requiredFields = [
            'first_name', 'last_name', 'date_of_birth', 'gender',
            'address_line_1', 'city', 'postal_code', 'country',
            'phone_number', 'occupation'
        ];

        $filledFields = 0;
        foreach ($requiredFields as $field) {
            if (!empty($this->$field)) {
                $filledFields++;
            }
        }

        return round(($filledFields / count($requiredFields)) * 100);
    }

    public function render()
    {
        return view('livewire.auth.profile-completion', [
            'completionPercentage' => $this->completionPercentage,
        ]);
    }
}
