<?php

namespace App\Livewire\Auth;

use App\Models\User;
use App\Models\TwoFactorAuth;
use App\Mail\TwoFactorVerificationMail;
use Illuminate\Auth\Events\Lockout;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use Livewire\Component;
use Livewire\Attributes\Validate;
use Livewire\Attributes\Layout;

#[Layout('components.layouts.auth')]
class Login extends Component
{
    public string $login = '';
    public string $password = '';
    public bool $remember = false;
    public bool $showTwoFactor = false;
    public string $twoFactorCode = '';
    public string $twoFactorMethod = '';
    public bool $isLockedOut = false;
    public int $remainingAttempts = 5;
    public int $lockoutTime = 0;

    protected $rules = [
        'login' => 'required|string',
        'password' => 'required|string',
    ];

    /**
     * Handle an incoming authentication request.
     */
    public function authenticate(): void
    {
        // Check if account is locked out
        if ($this->isAccountLockedOut()) {
            $this->isLockedOut = true;
            $this->lockoutTime = $this->getLockoutTime();
            return;
        }

        $this->ensureIsNotRateLimited();

        // Determine if login field is email or username
        $loginField = filter_var($this->login, FILTER_VALIDATE_EMAIL) ? 'email' : 'username';
        
        // Attempt authentication
        $credentials = [$loginField => $this->login, 'password' => $this->password];
        
        if (! Auth::attempt($credentials, $this->remember)) {
            $this->handleFailedLogin();
            return;
        }

        $user = Auth::user();
        
        // Check if user account is active or blocked
        if ($user->customer_status === 'blocked') {
            Auth::logout();
            $this->addError('login', 'Your account has been blocked. Please contact support.');
            return;
        }
        
        if (!$this->isUserAccountActive($user)) {
            Auth::logout();
            $errorMessage = $user->customer_status === 'suspended' 
                ? 'Your account is suspended. Please contact support.'
                : 'Your account is not active. Please contact support.';
            $this->addError('login', $errorMessage);
            return;
        }

        // Check if 2FA is required
        if ($this->isTwoFactorRequired($user)) {
            // Set pending 2FA session
            session(['pending_2fa_user_id' => $user->id]);
            $this->set2FASessionStartTime();
            
            // If email 2FA, send a new verification code
            if ($user->twoFactorAuth->method === 'email') {
                $this->sendEmailVerificationCode($user);
            }
            
            $this->showTwoFactor = true;
            $this->twoFactorMethod = $user->twoFactorAuth->method;
            return;
        }

        // No 2FA required, complete login immediately
        $this->handleSuccessfulLogin($user);
    }

    /**
     * Refresh the 2FA session to prevent timeout.
     */
    protected function refresh2FASession(): void
    {
        if (session()->has('pending_2fa_user_id')) {
            // Extend the session
            session()->put('pending_2fa_user_id', session('pending_2fa_user_id'));
        }
    }

    /**
     * Handle two-factor authentication.
     */
    public function verifyTwoFactor(): void
    {
        $this->validate([
            'twoFactorCode' => 'required|string|size:6'
        ]);

        // Validate 2FA session is still valid
        if (!$this->isPending2FASessionValid()) {
            $this->resetLogin();
            $this->addError('twoFactorCode', 'Two-factor authentication session expired. Please login again.');
            return;
        }

        $user = Auth::user();
        
        if (!$this->verifyTwoFactorCode($user, $this->twoFactorCode)) {
            $this->addError('twoFactorCode', 'Invalid two-factor authentication code.');
            return;
        }

        $this->completeLoginAfter2FA($user);
    }

    /**
     * Handle successful login (initial authentication).
     */
    protected function handleSuccessfulLogin(User $user): void
    {
        // Clear rate limiting
        RateLimiter::clear($this->throttleKey());
        
        // Set session flash
        session()->flash('login_success', true);
        
        // Check for IP change and send notification
        $currentIp = request()->ip();
        $lastLoginIp = $user->last_login_ip;
        
        if ($lastLoginIp && $lastLoginIp !== $currentIp) {
            // Send IP change notification
            $notificationService = app(\App\Services\NotificationService::class);
            $notificationService->notifyLoginIpChange($user, $currentIp);
        }
        
        // Update user activity
        $user->updateLastLogin();
        $user->updateLastActivity();
        $user->update(['last_login_ip' => $currentIp]);

        // Log successful login
        $this->logActivity($user, 'User logged in successfully');
        
        // Regenerate session for security
        Session::regenerate();
        
        // Redirect to dashboard
        $this->redirect(route('dashboard'), navigate: true);
    }

    /**
     * Complete login after successful 2FA verification.
     */
protected function completeLoginAfter2FA(User $user): void
    {
        // Clear pending 2FA session
        session()->forget(['pending_2fa_user_id', 'pending_2fa_start_time']);
        
        // Check for IP change and send notification
        $currentIp = request()->ip();
        $lastLoginIp = $user->last_login_ip;
        
        if ($lastLoginIp && $lastLoginIp !== $currentIp) {
            // Send IP change notification
            $notificationService = app(\App\Services\NotificationService::class);
            $notificationService->notifyLoginIpChange($user, $currentIp);
        }
        
        // Update user activity
        $user->updateLastLogin();
        $user->updateLastActivity();
        $user->update(['last_login_ip' => $currentIp]);
        
        // Set session flash for welcome message (since this is the actual completion of login)
        session()->flash('login_success', true);
        
        // Regenerate session for security
        Session::regenerate();
        
        // Redirect to dashboard
        $this->redirect(route('dashboard'), navigate: true);
    }

    /**
     * Handle failed login.
     */
    protected function handleFailedLogin(): void
    {
        // Increment failed attempts
        RateLimiter::hit($this->throttleKey());

        // Log failed login attempt
        $this->logActivity(null, 'Failed login attempt', [
            'login' => $this->login,
            'ip' => request()->ip(),
            'user_agent' => request()->userAgent()
        ]);

        // Check if account should be locked
        if (RateLimiter::tooManyAttempts($this->throttleKey(), 5)) {
            $this->lockAccount();
        }

        throw ValidationException::withMessages([
            'login' => __('auth.failed'),
        ]);
    }

    /**
     * Check if user account is active.
     */
    protected function isUserAccountActive(User $user): bool
    {
        return $user->customer_status === 'active' || $user->customer_status === 'transaction_declined';
    }

    /**
     * Check if two-factor authentication is required.
     */
    protected function isTwoFactorRequired(User $user): bool
    {
        return $user->twoFactorAuth && $user->twoFactorAuth->enabled;
    }

    /**
     * Verify two-factor authentication code.
     */
    protected function verifyTwoFactorCode(User $user, string $code): bool
    {
        $twoFactor = $user->twoFactorAuth;
        
        if (!$twoFactor || !$twoFactor->enabled) {
            return false;
        }

        switch ($twoFactor->method) {
            case 'totp':
                return $this->verifyTotpCode($twoFactor, $code);
            case 'email':
                return $this->verifyEmailCode($twoFactor, $code);
            case 'passkey':
                return $this->verifyPasskey($twoFactor, $code);
            default:
                return false;
        }
    }

    /**
     * Verify TOTP code.
     */
    protected function verifyTotpCode(TwoFactorAuth $twoFactor, string $code): bool
    {
        try {
            // Use Google2FA library to verify TOTP code
            $google2fa = new \PragmaRX\Google2FA\Google2FA();
            
            // Get the secret key from the 2FA record
            $secret = $twoFactor->secret_key;
            
            if (!$secret) {
                return false;
            }
            
            // Verify the TOTP code
            $result = $google2fa->verifyKey($secret, $code, 2); // Allow 2 time steps for clock skew
            
            return $result;
        } catch (\Exception $e) {
            \Log::error('TOTP verification error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Verify email code.
     */
    protected function verifyEmailCode(TwoFactorAuth $twoFactor, string $code): bool
    {
        return $twoFactor->verification_code === $code && !$twoFactor->is_expired;
    }

    /**
     * Verify passkey.
     */
    protected function verifyPasskey(TwoFactorAuth $twoFactor, string $code): bool
    {
        // This would integrate with the Spatie Passkeys package
        return true; // Placeholder
    }

    /**
     * Check if account is locked out.
     */
    protected function isAccountLockedOut(): bool
    {
        return RateLimiter::tooManyAttempts($this->throttleKey(), 5);
    }

    /**
     * Get lockout time remaining.
     */
    protected function getLockoutTime(): int
    {
        return RateLimiter::availableIn($this->throttleKey());
    }

    /**
     * Lock account after too many failed attempts.
     */
    protected function lockAccount(): void
    {
        event(new Lockout(request()));
        
        // Log account lockout
        $this->logActivity(null, 'Account locked due to multiple failed attempts', [
            'login' => $this->login,
            'ip' => request()->ip()
        ]);
    }

    /**
     * Log activity using Spatie Activity Log.
     */
    protected function logActivity(?User $user, string $description, array $properties = []): void
    {
        if ($user) {
            activity()
                ->causedBy($user)
                ->log($description);
        } else {
            activity()
                ->withProperties($properties)
                ->log($description);
        }
    }

    /**
     * Ensure the authentication request is not rate limited.
     */
    protected function ensureIsNotRateLimited(): void
    {
        if (! RateLimiter::tooManyAttempts($this->throttleKey(), 5)) {
            return;
        }

        event(new Lockout(request()));

        $seconds = RateLimiter::availableIn($this->throttleKey());

        throw ValidationException::withMessages([
            'login' => __('auth.throttle', [
                'seconds' => $seconds,
                'minutes' => ceil($seconds / 60),
            ]),
        ]);
    }

    /**
     * Get the authentication rate limiting throttle key.
     */
    protected function throttleKey(): string
    {
        return Str::transliterate(Str::lower($this->login).'|'.request()->ip());
    }

    /**
     * Reset login form and return to initial state.
     */
    public function resetLogin(): void
    {
        // Clear pending 2FA session
        session()->forget('pending_2fa_user_id');
        session()->forget('pending_2fa_start_time');
        
        // Log out user if they were partially authenticated
        if (Auth::check()) {
            Auth::logout();
        }
        
        // Reset component state
        $this->reset(['showTwoFactor', 'twoFactorCode', 'twoFactorMethod']);
        $this->resetValidation();
    }

    /**
     * Cancel 2FA and return to login form.
     */
    public function cancelTwoFactor(): void
    {
        $this->resetLogin();
    }

    /**
     * Check if the 2FA session has timed out.
     */
    protected function is2FASessionTimedOut(): bool
    {
        // Check if 2FA session is older than 10 minutes
        if (session()->has('pending_2fa_user_id')) {
            $sessionStart = session('pending_2fa_start_time', 0);
            $timeout = 10 * 60; // 10 minutes in seconds
            
            if (time() - $sessionStart > $timeout) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Set the 2FA session start time.
     */
    protected function set2FASessionStartTime(): void
    {
        session(['pending_2fa_start_time' => time()]);
    }

    /**
     * Check if the pending 2FA session is still valid.
     */
    protected function isPending2FASessionValid(): bool
    {
        if (!session()->has('pending_2fa_user_id')) {
            return false;
        }
        
        // Check if session has timed out
        if ($this->is2FASessionTimedOut()) {
            return false;
        }
        
        $pendingUserId = session('pending_2fa_user_id');
        $user = Auth::user();
        
        // Check if user is still authenticated and matches pending session
        if (!$user || $user->id !== $pendingUserId) {
            return false;
        }
        
        // Check if 2FA is still required
        if (!$this->isTwoFactorRequired($user)) {
            return false;
        }
        
        return true;
    }

    /**
     * Validate 2FA session on each request.
     */
    public function updated($propertyName)
    {
        // If we're showing 2FA, validate the session is still valid
        if ($this->showTwoFactor && $propertyName !== 'twoFactorCode') {
            if (!$this->isPending2FASessionValid()) {
                $this->resetLogin();
            }
        }
    }

    public function mount()
    {
        $this->login = old('login') ?? '';
        $this->remember = old('remember') ?? false;
        
        // Check if user is already authenticated
        if (Auth::check()) {
            $user = Auth::user();
            
            // If user is authenticated but 2FA is required and not completed
            if ($this->isTwoFactorRequired($user)) {
                // Check if we have a pending 2FA session
                if (session()->has('pending_2fa_user_id') && session('pending_2fa_user_id') === $user->id) {
                    // Check if session has timed out
                    if ($this->is2FASessionTimedOut()) {
                        // Session expired, clear everything and require full login
                        $this->resetLogin();
                        return;
                    }
                    
                    // Valid pending 2FA session, show 2FA form
                    $this->showTwoFactor = true;
                    $this->twoFactorMethod = $user->twoFactorAuth->method;
                    return;
                }
                
                // If no pending 2FA session, log them out and require full login
                Auth::logout();
                session()->forget(['pending_2fa_user_id', 'pending_2fa_start_time']);
            } else {
                // User is fully authenticated, redirect to dashboard
                $this->redirect(route('dashboard'), navigate: true);
            }
        }
    }

    /**
     * Send email verification code for 2FA.
     */
    protected function sendEmailVerificationCode(User $user): void
    {
        try {
            $twoFactor = $user->twoFactorAuth;
            $code = $twoFactor->generateEmailCode();
            $expiresAt = $twoFactor->verification_code_expires_at->format('Y-m-d H:i:s T');
            
            Mail::to($user->email)->send(new TwoFactorVerificationMail($code, $user->name, $expiresAt));
            
            // Log the email sent
            $this->logActivity($user, '2FA verification email sent');
            
        } catch (\Exception $e) {
            \Log::error('Failed to send 2FA verification email: ' . $e->getMessage());
            
            // Log the failure
            $this->logActivity($user, 'Failed to send 2FA verification email', [
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Resend email verification code for 2FA.
     */
    public function resendEmailCode(): void
    {
        if (!$this->isPending2FASessionValid()) {
            $this->resetLogin();
            $this->addError('twoFactorCode', 'Two-factor authentication session expired. Please login again.');
            return;
        }

        $user = Auth::user();
        
        if ($user && $user->twoFactorAuth && $user->twoFactorAuth->method === 'email') {
            $this->sendEmailVerificationCode($user);
            
            // Show success message
            session()->flash('message', 'Verification code has been resent to your email.');
        }
    }

    public function render()
    {
        return view('livewire.auth.login');
    }
}
