<?php

namespace App\Livewire\Auth;

use App\Models\User;
use Illuminate\Support\Facades\Mail;
use Livewire\Component;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Validate;
use App\Mail\PasswordResetCodeMail;
use Flux;

#[Layout('components.layouts.auth')]
class ForgotPassword extends Component
{
    #[Validate('required|email|exists:users,email')]
    public string $email = '';

    #[Validate('required|string|size:6')]
    public string $resetCode = '';

    public bool $codeSent = false;
    public bool $isLoading = false;
    public int $resendCooldown = 0;
    public int $attempts = 0;
    public int $maxAttempts = 3;
    public string $status = '';
    


    public function sendPasswordResetCode()
    {
        

        $this->validate(['email']);

        if ($this->resendCooldown > 0) {
            Flux::toast('Please wait before requesting another code.', variant: 'warning');
            return;
        }

        $user = User::where('email', $this->email)->first();
        
        if (!$user) {
            $this->addError('email', 'No account found with this email address.');
            return;
        }

        try {
            $code = $user->generatePasswordResetCode();
            
            // Send email with code
            Mail::to($user->email)->send(new PasswordResetCodeMail($code, $user));

            $this->codeSent = true;
            $this->resendCooldown = 60; // 60 seconds cooldown
            $this->status = 'Password reset code sent to your email.';
            
            Flux::toast('Password reset code sent to your email.', variant: 'success');
            
            // Start countdown
            $this->dispatch('start-resend-countdown');

        } catch (\Exception $e) {
            Flux::toast('Failed to send reset code. Please try again.', variant: 'danger');
            \Log::error('Password reset code send error: ' . $e->getMessage());
        }
    }

    public function verifyResetCode()
    {
        
        $this->validate(['resetCode']);

        $user = User::where('email', $this->email)->first();
        
        if (!$user) {
            $this->addError('email', 'No account found with this email address.');
            return;
        }

        if ($this->attempts >= $this->maxAttempts) {
            Flux::toast('Too many failed attempts. Please request a new code.', variant: 'danger');
            return;
        }

        $this->isLoading = true;

        if ($user->verifyPasswordResetCode($this->resetCode)) {
            // Store user ID in session for password reset
            session(['password_reset_user_id' => $user->id]);
            
            Flux::toast('Code verified! Redirecting to reset password...', variant: 'success');
            $this->redirect(route('password.reset'), navigate: true);
        } else {
            $this->attempts++;
            $this->resetCode = '';
            
            if ($this->attempts >= $this->maxAttempts) {
                Flux::toast('Too many failed attempts. Please request a new code.', variant: 'danger');
            } else {
                $remaining = $this->maxAttempts - $this->attempts;
                Flux::toast("Invalid code. {$remaining} attempts remaining.", variant: 'danger');
            }
        }

        $this->isLoading = false;
    }

    public function resendCode()
    {
        $this->sendPasswordResetCode();
    }

    public function resetForm()
    {
        $this->reset(['email', 'codeSent', 'status', 'resetCode', 'attempts']);
        $this->resetErrorBag();
    }

    public function render()
    {
        return view('livewire.auth.forgot-password');
    }
}
