<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Foundation\Http\Middleware\PreventRequestsDuringMaintenance as Middleware;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class PreventRequestsDuringMaintenance extends Middleware
{
    /**
     * The URIs that should be reachable while maintenance mode is enabled.
     *
     * @var array<int, string>
     */
    protected $except = [
        'admin/*',
        'admin',
        'admin/login',
        'admin/logout',
    ];

    /**
     * Handle an incoming request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure  $next
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function handle($request, Closure $next)
    {
        // Check if maintenance mode is enabled
        if ($this->app->isDownForMaintenance()) {
            // Allow admin routes to pass through
            if ($this->isAdminRoute($request)) {
                return $next($request);
            }
            
            // Allow admin users to access admin panel even during maintenance
            if ($this->isAdminUser($request)) {
                return $next($request);
            }
        }

        return parent::handle($request, $next);
    }

    /**
     * Check if the request is for an admin route
     */
    protected function isAdminRoute(Request $request): bool
    {
        $path = $request->path();
        
        // Check if path starts with admin/
        if (str_starts_with($path, 'admin/')) {
            return true;
        }
        
        // Check if it's the admin root
        if ($path === 'admin') {
            return true;
        }
        
        // Check route name if available
        $routeName = $request->route()?->getName();
        if ($routeName && str_starts_with($routeName, 'admin.')) {
            return true;
        }
        
        return false;
    }

    /**
     * Check if the user is an admin user
     */
    protected function isAdminUser(Request $request): bool
    {
        if (!auth()->check()) {
            return false;
        }

        $user = auth()->user();
        
        // Check if user has admin or super-admin role
        return $user->hasRole(['admin', 'super-admin']);
    }
}
