<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Symfony\Component\HttpFoundation\Response;

class ErrorHandlingMiddleware
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        try {
            $response = $next($request);
            
            // Log successful requests for monitoring
            if ($response->getStatusCode() >= 200 && $response->getStatusCode() < 300) {
                $this->logSuccessfulRequest($request, $response);
            }
            
            return $response;
        } catch (\Exception $e) {
            // Log the error with context
            $this->logError($request, $e);
            
            // Re-throw the exception to let Laravel's error handler deal with it
            throw $e;
        }
    }

    /**
     * Log successful requests for monitoring purposes.
     */
    private function logSuccessfulRequest(Request $request, Response $response): void
    {
        // Only log important requests to avoid log spam
        $importantRoutes = [
            'transfer',
        ];

        $routeName = $request->route()?->getName();
        
        if ($routeName && in_array($routeName, $importantRoutes)) {
            Log::info('Important request completed', [
                'route' => $routeName,
                'method' => $request->method(),
                'url' => $request->fullUrl(),
                'user_id' => auth()->id(),
                'status_code' => $response->getStatusCode(),
                'ip' => $request->ip(),
                'user_agent' => $request->userAgent(),
            ]);
        }
    }

    /**
     * Log errors with detailed context.
     */
    private function logError(Request $request, \Exception $e): void
    {
        $context = [
            'exception' => get_class($e),
            'message' => $e->getMessage(),
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $e->getTraceAsString(),
            'request' => [
                'method' => $request->method(),
                'url' => $request->fullUrl(),
                'route' => $request->route()?->getName(),
                'ip' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'user_id' => auth()->id(),
                'input' => $this->sanitizeInput($request->all()),
            ],
        ];

        // Log different types of errors with appropriate levels
        if ($e instanceof \Illuminate\Validation\ValidationException) {
            Log::warning('Validation error', $context);
        } elseif ($e instanceof \Illuminate\Auth\AuthenticationException) {
            Log::info('Authentication error', $context);
        } elseif ($e instanceof \Illuminate\Database\Eloquent\ModelNotFoundException) {
            Log::warning('Model not found', $context);
        } else {
            Log::error('Application error', $context);
        }
    }

    /**
     * Sanitize input data to remove sensitive information.
     */
    private function sanitizeInput(array $input): array
    {
        $sensitiveFields = [
            'password',
            'password_confirmation',
            'current_password',
            'pin',
            'pin_confirmation',
            'card_number',
            'cvv',
            'ssn',
            'social_security_number',
        ];

        foreach ($sensitiveFields as $field) {
            if (isset($input[$field])) {
                $input[$field] = '[REDACTED]';
            }
        }

        return $input;
    }
}
