<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;

class ContactController extends Controller
{
    public function index()
    {
        return view('contact');
    }

    public function store(Request $request)
    {
        // Validate the form data
        $validator = Validator::make($request->all(), [
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'nullable|string|max:20',
            'subject' => 'required|string|max:255',
            'message' => 'required|string|max:2000',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            // Prepare email data
            $emailData = [
                'first_name' => $request->first_name,
                'last_name' => $request->last_name,
                'email' => $request->email,
                'phone' => $request->phone,
                'subject' => $request->subject,
                'message' => $request->message,
                'bank_name' => \App\Helpers\SettingsHelper::bankName(),
                'bank_email' => \App\Helpers\SettingsHelper::bankEmail(),
            ];

            // Send email to bank
            Mail::send('emails.contact-form', $emailData, function ($message) use ($emailData) {
                $message->to($emailData['bank_email'])
                    ->subject('New Contact Form Submission: ' . $emailData['subject'])
                    ->from($emailData['email'], $emailData['first_name'] . ' ' . $emailData['last_name']);
            });

            // Send confirmation email to customer
            $emailData['bank_phone'] = \App\Helpers\SettingsHelper::bankPhone();
            Mail::send('emails.contact-confirmation', $emailData, function ($message) use ($emailData) {
                $message->to($emailData['email'])
                    ->subject('Thank you for contacting ' . $emailData['bank_name'])
                    ->from($emailData['bank_email'], $emailData['bank_name']);
            });

            return redirect()->back()->with('success', 'Thank you for your message! We will get back to you soon.');

        } catch (\Exception $e) {
            // Log the error
            \Log::error('Contact form submission failed: ' . $e->getMessage());
            
            return redirect()->back()
                ->with('error', 'Sorry, there was an error sending your message. Please try again later.')
                ->withInput();
        }
    }
}
