<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\UserProfile;
use App\Models\Code;
use App\Models\Account;
use App\Models\Transaction;
use App\Models\KycDocument;
use App\Models\ComplianceLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;
use Carbon\Carbon;
use Spatie\Permission\Models\Role;

class UserController extends Controller
{
    /**
     * Display a listing of users with search, filters, and pagination.
     */
    public function index(Request $request)
    {
        $query = User::with(['profile', 'roles', 'accounts'])
            ->withCount(['transactions', 'kycDocuments', 'complianceLogs'])
            ->whereDoesntHave('roles', function ($q) {
                $q->whereIn('name', ['admin', 'super-admin']);
            });

        // Search functionality
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('customer_id', 'like', "%{$search}%")
                  ->orWhere('username', 'like', "%{$search}%")
                  ->orWhereHas('profile', function ($profileQuery) use ($search) {
                      $profileQuery->where('first_name', 'like', "%{$search}%")
                                   ->orWhere('last_name', 'like', "%{$search}%")
                                   ->orWhere('phone_number', 'like', "%{$search}%");
                  });
            });
        }

        // Filter by customer status
        if ($request->filled('status')) {
            $query->where('customer_status', $request->status);
        }

        // Filter by customer type
        if ($request->filled('type')) {
            $query->where('customer_type', $request->type);
        }

        // Filter by role
        if ($request->filled('role')) {
            $query->role($request->role);
        }

        // Filter by KYC status
        if ($request->filled('kyc_status')) {
            $query->whereHas('profile', function ($q) use ($request) {
                $q->where('kyc_status', $request->kyc_status);
            });
        }

        // Filter by registration date
        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        // Sort functionality
        $sortBy = $request->get('sort_by', 'created_at');
        $sortDirection = $request->get('sort_direction', 'desc');
        
        $allowedSorts = ['name', 'email', 'customer_id', 'customer_status', 'created_at', 'last_login_at'];
        if (in_array($sortBy, $allowedSorts)) {
            $query->orderBy($sortBy, $sortDirection);
        }

        $users = $query->paginate(20)->withQueryString();

        // Get filter options
        $filterOptions = [
            'statuses' => ['active', 'inactive', 'suspended', 'pending_verification', 'closed', 'transaction_declined', 'blocked'],
            'types' => ['individual', 'business', 'corporate', 'premium'],
            'roles' => Role::all()->pluck('name', 'name'),
            'kyc_statuses' => ['pending', 'verified', 'rejected', 'under_review'],
        ];

        return view('admin.users.index', compact('users', 'filterOptions'));
    }

    /**
     * Show the form for creating a new user.
     */
    public function create()
    {
        $roles = Role::where('name', '!=', 'super-admin')->where('name', '!=', 'admin')->get();
        $customerTypes = ['individual', 'business', 'corporate', 'premium'];
        $customerStatuses = ['active', 'inactive', 'pending_verification', 'transaction_declined', 'blocked'];
        
        return view('admin.users.create', compact('roles', 'customerTypes', 'customerStatuses'));
    }

    /**
     * Store a newly created user in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'username' => 'nullable|string|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'customer_type' => 'required|in:individual,business,corporate,premium',
            'customer_status' => 'required|in:active,inactive,pending_verification,transaction_declined,blocked',
            'role' => 'required|exists:roles,name',
            'pin' => 'nullable|string|min:4|max:6',
            
            // Profile fields
            'first_name' => 'required|string|max:100',
            'last_name' => 'required|string|max:100',
            'middle_name' => 'nullable|string|max:100',
            'date_of_birth' => 'required|date|before:today',
            'gender' => 'nullable|in:male,female,other,prefer_not_to_say',
            'marital_status' => 'nullable|in:single,married,divorced,widowed,separated',
            'occupation' => 'required|string|max:255',
            'annual_income' => 'nullable|numeric|min:0',
            'address_line_1' => 'required|string|max:255',
            'city' => 'required|string|max:100',
            'state_province' => 'nullable|string|max:100',
            'postal_code' => 'required|string|max:20',
            'country' => 'required|string|max:100',
            'phone_number' => 'required|string|max:20',
            'emergency_contact_name' => 'nullable|string|max:255',
            'emergency_contact_phone' => 'nullable|string|max:20',
            'emergency_contact_relationship' => 'nullable|string|max:100',
            'preferred_language' => 'nullable|string|max:10',
            'timezone' => 'nullable|string|max:50',
            'notes' => 'nullable|string|max:1000',
            
            // Code fields
            'cot_code' => 'nullable|string|max:50',
            'cot_status' => 'in:0,1',
            'cot_msg' => 'nullable|string|max:500',
            'tax_code' => 'nullable|string|max:50',
            'tax_status' => 'in:0,1',
            'tax_msg' => 'nullable|string|max:500',
            'imf_code' => 'nullable|string|max:50',
            'imf_status' => 'in:0,1',
            'imf_msg' => 'nullable|string|max:500',
            'allow_local' => 'in:0,1',
            'allow_international' => 'in:0,1',
            'kyc_status' => 'nullable|in:pending,verified,rejected,under_review',
            'aml_status' => 'nullable|in:pending,clear,flagged,blocked',
            'compliance_status' => 'nullable|in:compliant,non_compliant,pending_review,under_investigation',
        ]);

        DB::beginTransaction();

        try {
            // Create user
            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'username' => $request->username,
                'password' => Hash::make($request->password),
                'customer_type' => $request->customer_type,
                'customer_status' => $request->customer_status,
                'pin' => $request->pin,
                'email_verified_at' => now(),
                'account_activated_at' => $request->customer_status === 'active' ? now() : null,
            ]);

            // Assign role
            $user->assignRole($request->role);

            // Create profile
            $user->profile()->create([
                'first_name' => $request->first_name,
                'last_name' => $request->last_name,
                'middle_name' => $request->middle_name,
                'date_of_birth' => $request->date_of_birth,
                'gender' => $request->gender,
                'marital_status' => $request->marital_status,
                'occupation' => $request->occupation,
                'annual_income' => $request->annual_income,
                'address_line_1' => $request->address_line_1,
                'city' => $request->city,
                'state_province' => $request->state_province,
                'postal_code' => $request->postal_code,
                'country' => $request->country,
                'phone_number' => $request->phone_number,
                'emergency_contact_name' => $request->emergency_contact_name,
                'emergency_contact_phone' => $request->emergency_contact_phone,
                'emergency_contact_relationship' => $request->emergency_contact_relationship,
                'preferred_language' => $request->preferred_language ?: 'en',
                'timezone' => $request->timezone ?: 'UTC',
                'notes' => $request->notes,
                'kyc_status' => $request->kyc_status ?: 'pending',
                'aml_status' => $request->aml_status ?: 'pending',
                'compliance_status' => $request->compliance_status ?: 'pending_review',
            ]);

            // Create code record
            $user->code()->create([
                'cot_code' => $request->cot_code,
                'cot_status' => $request->cot_status === '1',
                'cot_msg' => $request->cot_msg,
                'tax_code' => $request->tax_code,
                'tax_status' => $request->tax_status === '1',
                'tax_msg' => $request->tax_msg,
                'imf_code' => $request->imf_code,
                'imf_status' => $request->imf_status === '1',
                'imf_msg' => $request->imf_msg,
                'allow_local' => $request->allow_local === '1',
                'allow_international' => $request->allow_international === '1',
            ]);

            // Log the creation
            activity()
                ->performedOn($user)
                ->log("User created by admin: {$user->name}");

            DB::commit();

            return redirect()->route('admin.users.show', $user)
                ->with('success', 'User created successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withInput()
                ->with('error', 'Failed to create user: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified user.
     */
    public function show(User $user)
    {
        $user->load([
            'profile',
            'roles',
            'accounts',
            'transactions' => function ($query) {
                $query->latest()->limit(10);
            },
            'kycDocuments' => function ($query) {
                $query->latest();
            },
            'complianceLogs' => function ($query) {
                $query->latest()->limit(10);
            },
            'beneficiaries',
            'virtualCards',
            'code'
        ]);

        // Get user statistics
        $userStats = [
            'total_accounts' => $user->accounts()->count(),
            'active_accounts' => $user->accounts()->where('status', 'active')->count(),
            'total_balance' => $user->getTotalBalance(),
            'total_transactions' => $user->transactions()->count(),
            'pending_transactions' => $user->transactions()->where('status', 'pending')->count(),
            'kyc_documents' => $user->kycDocuments()->count(),
            'pending_kyc' => $user->kycDocuments()->where('status', 'pending')->count(),
            'compliance_flags' => $user->complianceLogs()->where('status', 'pending')->count(),
            'beneficiaries' => $user->beneficiaries()->where('active', true)->count(),
            'virtual_cards' => $user->virtualCards()->where('status', 'active')->count(),
        ];

        // Get recent activity
        $recentActivity = $this->getUserRecentActivity($user);

        return view('admin.users.show', compact('user', 'userStats', 'recentActivity'));
    }

    /**
     * Show the form for editing the specified user.
     */
    public function edit(User $user)
    {
        $user->load('profile', 'roles', 'code');
        
        $roles = Role::where('name', '!=', 'super-admin')->where('name', '!=', 'admin')->get();
        $customerTypes = ['individual', 'business', 'corporate', 'premium'];
        $customerStatuses = ['active', 'inactive', 'suspended', 'pending_verification', 'closed', 'transaction_declined', 'blocked'];
        
        return view('admin.users.edit', compact('user', 'roles', 'customerTypes', 'customerStatuses'));
    }

    /**
     * Update the specified user in storage.
     */
    public function update(Request $request, User $user)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => ['required', 'string', 'email', 'max:255', Rule::unique('users')->ignore($user->id)],
            'username' => ['nullable', 'string', 'max:255', Rule::unique('users')->ignore($user->id)],
            'password' => 'nullable|string|min:8|confirmed',
            'customer_type' => 'required|in:individual,business,corporate,premium',
            'customer_status' => 'required|in:active,inactive,suspended,pending_verification,closed,transaction_declined,blocked',
            'role' => 'required|exists:roles,name',
            'pin' => 'nullable|string|min:4|max:6',
            
            // Profile fields
            'first_name' => 'required|string|max:100',
            'last_name' => 'required|string|max:100',
            'middle_name' => 'nullable|string|max:100',
            'date_of_birth' => 'required|date|before:today',
            'gender' => 'nullable|in:male,female,other,prefer_not_to_say',
            'marital_status' => 'nullable|in:single,married,divorced,widowed,separated',
            'occupation' => 'required|string|max:255',
            'annual_income' => 'nullable|numeric|min:0',
            'address_line_1' => 'required|string|max:255',
            'city' => 'required|string|max:100',
            'state_province' => 'nullable|string|max:100',
            'postal_code' => 'required|string|max:20',
            'country' => 'required|string|max:100',
            'phone_number' => 'required|string|max:20',
            'emergency_contact_name' => 'nullable|string|max:255',
            'emergency_contact_phone' => 'nullable|string|max:20',
            'emergency_contact_relationship' => 'nullable|string|max:100',
            'preferred_language' => 'nullable|string|max:10',
            'timezone' => 'nullable|string|max:50',
            'notes' => 'nullable|string|max:1000',
            
            // Code fields
            'cot_code' => 'nullable|string|max:50',
            'cot_status' => 'in:0,1',
            'cot_msg' => 'nullable|string|max:500',
            'tax_code' => 'nullable|string|max:50',
            'tax_status' => 'in:0,1',
            'tax_msg' => 'nullable|string|max:500',
            'imf_code' => 'nullable|string|max:50',
            'imf_status' => 'in:0,1',
            'imf_msg' => 'nullable|string|max:500',
            'allow_local' => 'in:0,1',
            'allow_international' => 'in:0,1',
            'kyc_status' => 'nullable|in:pending,verified,rejected,under_review',
            'aml_status' => 'nullable|in:pending,clear,flagged,blocked',
            'compliance_status' => 'nullable|in:compliant,non_compliant,pending_review,under_investigation',
        ]);

        DB::beginTransaction();

        try {
            // Update user
            $userData = [
                'name' => $request->name,
                'email' => $request->email,
                'username' => $request->username,
                'customer_type' => $request->customer_type,
                'customer_status' => $request->customer_status,
            ];

            if ($request->filled('password')) {
                $userData['password'] = Hash::make($request->password);
                $userData['password_changed_at'] = now();
            }

            if ($request->filled('pin')) {
                $userData['pin'] = $request->pin;
            }

            if ($request->customer_status === 'active' && !$user->account_activated_at) {
                $userData['account_activated_at'] = now();
            }

            $user->update($userData);

            // Update role
            $user->syncRoles([$request->role]);

            // Update profile
            $user->profile()->updateOrCreate(
                ['user_id' => $user->id],
                [
                    'first_name' => $request->first_name,
                    'last_name' => $request->last_name,
                    'middle_name' => $request->middle_name,
                    'date_of_birth' => $request->date_of_birth,
                    'gender' => $request->gender,
                    'marital_status' => $request->marital_status,
                    'occupation' => $request->occupation,
                    'annual_income' => $request->annual_income,
                    'address_line_1' => $request->address_line_1,
                    'city' => $request->city,
                    'state_province' => $request->state_province,
                    'postal_code' => $request->postal_code,
                    'country' => $request->country,
                    'phone_number' => $request->phone_number,
                    'emergency_contact_name' => $request->emergency_contact_name,
                    'emergency_contact_phone' => $request->emergency_contact_phone,
                    'emergency_contact_relationship' => $request->emergency_contact_relationship,
                    'preferred_language' => $request->preferred_language ?: 'en',
                    'timezone' => $request->timezone ?: 'UTC',
                    'kyc_status' => $request->kyc_status ?: 'pending',
                    'aml_status' => $request->aml_status ?: 'pending',
                    'compliance_status' => $request->compliance_status ?: 'pending_review',
                    'notes' => $request->notes,
                ]
            );

            // Update code only if user has filled in any values
            $codeData = [];
            
            if ($request->filled('cot_code')) {
                $codeData['cot_code'] = $request->cot_code;
                $codeData['cot_status'] = $request->cot_status === '1';
                $codeData['cot_msg'] = $request->cot_msg;
            }
            
            if ($request->filled('tax_code')) {
                $codeData['tax_code'] = $request->tax_code;
                $codeData['tax_status'] = $request->tax_status === '1';
                $codeData['tax_msg'] = $request->tax_msg;
            }
            
            if ($request->filled('imf_code')) {
                $codeData['imf_code'] = $request->imf_code;
                $codeData['imf_status'] = $request->imf_status === '1';
                $codeData['imf_msg'] = $request->imf_msg;
            }
            
            // Always update transfer permissions if provided
            if ($request->has('allow_local')) {
                $codeData['allow_local'] = $request->allow_local === '1';
            }
            if ($request->has('allow_international')) {
                $codeData['allow_international'] = $request->allow_international === '1';
            }
            
            // Only update/create if there's data to save
            if (!empty($codeData)) {
                $user->code()->updateOrCreate(
                    ['user_id' => $user->id],
                    $codeData
                );
            }

            // Log the update
            activity()
                ->performedOn($user)
                ->log("User updated by admin: {$user->name}");

            DB::commit();

            return redirect()->route('admin.users.show', $user)
                ->with('success', 'User updated successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withInput()
                ->with('error', 'Failed to update user: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified user from storage.
     */
    public function destroy(User $user)
    {
        // Prevent deletion of super-admin users
        if ($user->isSuperAdmin()) {
            return back()->with('error', 'Cannot delete super-admin users.');
        }

        // Prevent deletion of users with active accounts
        if ($user->accounts()->where('status', 'active')->exists()) {
            return back()->with('error', 'Cannot delete user with active accounts. Please close all accounts first.');
        }

        DB::beginTransaction();

        try {
            // Log the deletion
            activity()
                ->performedOn($user)
                ->log("User deleted by admin: {$user->name}");

            // Soft delete the user (cascades to related models)
            $user->delete();

            DB::commit();

            return redirect()->route('admin.users.index')
                ->with('success', 'User deleted successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to delete user: ' . $e->getMessage());
        }
    }

    /**
     * Activate user account.
     */
    public function activate(User $user)
    {
        if ($user->activateAccount()) {
            return back()->with('success', 'User account activated successfully.');
        }

        return back()->with('error', 'Failed to activate user account.');
    }

    /**
     * Suspend user account.
     */
    public function suspend(Request $request, User $user)
    {
        $request->validate([
            'reason' => 'required|string|max:500'
        ]);

        if ($user->suspendAccount($request->reason)) {
            return back()->with('success', 'User account suspended successfully.');
        }

        return back()->with('error', 'Failed to suspend user account.');
    }

    /**
     * Reset user password.
     */
    public function resetPassword(Request $request, User $user)
    {
        $request->validate([
            'password' => 'required|string|min:8|confirmed'
        ]);

        $user->update([
            'password' => Hash::make($request->password),
            'password_changed_at' => now(),
        ]);

        // Log the password reset
        activity()
            ->performedOn($user)
            ->log("Password reset by admin for user: {$user->name}");

        return back()->with('success', 'Password reset successfully.');
    }

    /**
     * Get user's recent activity.
     */
    private function getUserRecentActivity(User $user)
    {
        $activities = collect();

        // Recent transactions
        $recentTransactions = $user->transactions()
            ->latest()
            ->limit(5)
            ->get()
            ->map(function ($transaction) {
                return [
                    'type' => 'transaction',
                    'title' => 'Transaction',
                    'description' => '$' . number_format($transaction->amount) . ' ' . $transaction->type,
                    'time' => $transaction->created_at,
                    'icon' => 'arrow-right-left',
                    'icon_color' => 'blue',
                ];
            });

        // Recent KYC documents
        $recentKyc = $user->kycDocuments()
            ->latest()
            ->limit(3)
            ->get()
            ->map(function ($kyc) {
                return [
                    'type' => 'kyc',
                    'title' => 'KYC Document',
                    'description' => $kyc->document_type . ' - ' . ucfirst($kyc->status),
                    'time' => $kyc->created_at,
                    'icon' => 'document-text',
                    'icon_color' => $kyc->status === 'approved' ? 'green' : ($kyc->status === 'rejected' ? 'red' : 'yellow'),
                ];
            });

        // Recent compliance logs
        $recentCompliance = $user->complianceLogs()
            ->latest()
            ->limit(3)
            ->get()
            ->map(function ($log) {
                return [
                    'type' => 'compliance',
                    'title' => 'Compliance Log',
                    'description' => $log->description,
                    'time' => $log->created_at,
                    'icon' => 'exclamation-triangle',
                    'icon_color' => 'red',
                ];
            });

        // Merge and sort all activities
        $activities = $activities
            ->merge($recentTransactions)
            ->merge($recentKyc)
            ->merge($recentCompliance)
            ->sortByDesc('time')
            ->take(10);

        return $activities;
    }

    /**
     * Export users to CSV.
     */
    public function export(Request $request)
    {
        $query = User::with(['profile', 'roles']);

        // Apply same filters as index
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('customer_id', 'like', "%{$search}%");
            });
        }

        if ($request->filled('status')) {
            $query->where('customer_status', $request->status);
        }

        if ($request->filled('type')) {
            $query->where('customer_type', $request->type);
        }

        $users = $query->get();

        $filename = 'users_export_' . now()->format('Y-m-d_H-i-s') . '.csv';

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function () use ($users) {
            $file = fopen('php://output', 'w');

            // CSV headers
            fputcsv($file, [
                'Customer ID',
                'Name',
                'Email',
                'Customer Type',
                'Status',
                'Role',
                'Phone',
                'Country',
                'Registration Date',
                'Last Login',
                'Total Balance',
                'KYC Status'
            ]);

            // CSV data
            foreach ($users as $user) {
                fputcsv($file, [
                    $user->customer_id,
                    $user->name,
                    $user->email,
                    $user->customer_type,
                    $user->customer_status,
                    $user->roles->first()->name ?? 'No Role',
                    $user->profile->phone_number ?? '',
                    $user->profile->country ?? '',
                    $user->created_at->format('Y-m-d'),
                    $user->last_login_at ? $user->last_login_at->format('Y-m-d H:i:s') : 'Never',
                    $user->getTotalBalance(),
                    $user->profile->kyc_status ?? 'Unknown'
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}
