<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Transaction;
use App\Models\Account;
use App\Models\User;
use App\Models\Beneficiary;
use App\Services\TransactionService;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Str;

class TransferController extends Controller
{
    protected $transactionService;

    public function __construct(TransactionService $transactionService)
    {
        $this->transactionService = $transactionService;
    }

    /**
     * Display a listing of transfers.
     */
    public function index(Request $request)
    {
        $query = Transaction::with(['user', 'fromAccount', 'toAccount', 'beneficiary'])
            ->where('type', 'transfer');

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('transaction_id', 'like', "%{$search}%")
                  ->orWhere('reference_number', 'like', "%{$search}%")
                  ->orWhere('external_reference', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhereHas('user', function ($userQuery) use ($search) {
                      $userQuery->where('name', 'like', "%{$search}%")
                               ->orWhere('email', 'like', "%{$search}%");
                  })
                  ->orWhereHas('fromAccount', function ($accountQuery) use ($search) {
                      $accountQuery->where('account_number', 'like', "%{$search}%");
                  })
                  ->orWhereHas('toAccount', function ($accountQuery) use ($search) {
                      $accountQuery->where('account_number', 'like', "%{$search}%");
                  })
                  ->orWhereHas('beneficiary', function ($beneficiaryQuery) use ($search) {
                      $beneficiaryQuery->where('name', 'like', "%{$search}%")
                                      ->orWhere('email', 'like', "%{$search}%");
                  });
            });
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('transfer_method')) {
            $query->where('transfer_method', $request->transfer_method);
        }

        if ($request->filled('transfer_speed')) {
            $query->where('transfer_speed', $request->transfer_speed);
        }

        if ($request->filled('currency')) {
            $query->where('currency', $request->currency);
        }

        if ($request->filled('user_id')) {
            $query->where('user_id', $request->user_id);
        }

        if ($request->filled('from_account_id')) {
            $query->where('from_account_id', $request->from_account_id);
        }

        if ($request->filled('to_account_id')) {
            $query->where('to_account_id', $request->to_account_id);
        }

        if ($request->filled('beneficiary_id')) {
            $query->where('beneficiary_id', $request->beneficiary_id);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        if ($request->filled('amount_min')) {
            $query->where('amount', '>=', $request->amount_min);
        }

        if ($request->filled('amount_max')) {
            $query->where('amount', '<=', $request->amount_max);
        }

        // Apply sorting
        $sortBy = $request->get('sort_by', 'created_at');
        $sortDirection = $request->get('sort_direction', 'desc');
        
        $allowedSortFields = ['created_at', 'transaction_id', 'amount', 'status', 'transfer_method', 'transfer_speed', 'estimated_arrival', 'actual_arrival'];
        if (in_array($sortBy, $allowedSortFields)) {
            $query->orderBy($sortBy, $sortDirection);
        }

        $transfers = $query->paginate(10)->withQueryString();

        // Get filter options
        $users = User::select('id', 'name', 'email')->get();
        $accounts = Account::select('id', 'account_number', 'account_name')->get();
        $beneficiaries = Beneficiary::select('id', 'name', 'email')->get();
        
        return view('admin.transfers.index', compact('transfers', 'users', 'accounts', 'beneficiaries'));
    }

    /**
     * Show the form for creating a new transfer.
     */
    public function create()
    {
        $users = User::select('id', 'name', 'email')->get();
        $accounts = Account::select('id', 'account_number', 'account_name')->get();
        $beneficiaries = Beneficiary::select('id', 'name', 'email')->get();
        
        return view('admin.transfers.create', compact('users', 'accounts', 'beneficiaries'));
    }

    /**
     * Store a newly created transfer.
     */
    public function store(Request $request)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
            'from_account_id' => 'required|exists:accounts,id',
            'to_account_id' => 'nullable|exists:accounts,id',
            'beneficiary_id' => 'nullable|exists:beneficiaries,id',
            'category' => 'required|string|max:100',
            'subcategory' => 'nullable|string|max:100',
            'amount' => 'required|numeric|min:0.01',
            'currency' => 'required|string|size:3',
            'exchange_rate' => 'nullable|numeric|min:0',
            'converted_amount' => 'nullable|numeric|min:0',
            'fee_amount' => 'nullable|numeric|min:0',
            'tax_amount' => 'nullable|numeric|min:0',
            'net_amount' => 'nullable|numeric|min:0',
            'description' => 'required|string|max:500',
            'notes' => 'nullable|string|max:1000',
            'status' => 'required|in:pending,processing,completed,failed,cancelled,reversed',
            'verification_status' => 'required|in:pending,verified,rejected',
            'transfer_method' => 'required|in:wire,ach,swift,sepa,instant,domestic',
            'transfer_speed' => 'required|in:instant,same_day,next_day,standard',
            'estimated_arrival' => 'nullable|date|after:now',
            'external_reference' => 'nullable|string|max:255',
            'ip_address' => 'nullable|ip',
        ]);

        try {
            // Prepare transfer data
            $transferData = $request->all();
            $transferData['type'] = 'transfer';

            // Validate transfer data using service
            $validationErrors = $this->transactionService->validateTransactionData($transferData);
            if (!empty($validationErrors)) {
                return redirect()->back()
                    ->withErrors(['validation' => $validationErrors])
                    ->withInput();
            }

            // Create transfer using service
            $transfer = $this->transactionService->createTransaction($transferData);

            // Log the transfer creation
            activity()
                ->performedOn($transfer)
                ->log("Transfer created: {$transfer->transaction_id} - {$transfer->transfer_method}");

            return redirect()->route('admin.transfers.show', $transfer)
                ->with('success', 'Transfer created successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to create transfer: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display the specified transfer.
     */
    public function show(Transaction $transfer)
    {
        // Ensure this is a transfer transaction
        if ($transfer->type !== 'transfer') {
            abort(404, 'Transfer not found.');
        }

        $transfer->load(['user', 'fromAccount', 'toAccount', 'beneficiary']);
        
        return view('admin.transfers.show', compact('transfer'));
    }

    /**
     * Show the form for editing the specified transfer.
     */
    public function edit(Transaction $transfer)
    {
        // Ensure this is a transfer transaction
        if ($transfer->type !== 'transfer') {
            abort(404, 'Transfer not found.');
        }

        $users = User::select('id', 'name', 'email')->get();
        $accounts = Account::select('id', 'account_number', 'account_name')->get();
        $beneficiaries = Beneficiary::select('id', 'name', 'email')->get();
        
        return view('admin.transfers.edit', compact('transfer', 'users', 'accounts', 'beneficiaries'));
    }

    /**
     * Update the specified transfer.
     */
    public function update(Request $request, Transaction $transfer)
    {
        // Ensure this is a transfer transaction
        if ($transfer->type !== 'transfer') {
            abort(404, 'Transfer not found.');
        }

        $request->validate([
            'user_id' => 'required|exists:users,id',
            'from_account_id' => 'required|exists:accounts,id',
            'to_account_id' => 'nullable|exists:accounts,id',
            'beneficiary_id' => 'nullable|exists:beneficiaries,id',
            'category' => 'required|string|max:100',
            'subcategory' => 'nullable|string|max:100',
            'amount' => 'required|numeric|min:0.01',
            'currency' => 'required|string|size:3',
            'exchange_rate' => 'nullable|numeric|min:0',
            'converted_amount' => 'nullable|numeric|min:0',
            'fee_amount' => 'nullable|numeric|min:0',
            'tax_amount' => 'nullable|numeric|min:0',
            'net_amount' => 'nullable|numeric|min:0',
            'description' => 'required|string|max:500',
            'notes' => 'nullable|string|max:1000',
            'status' => 'required|in:pending,processing,completed,failed,cancelled,reversed',
            'verification_status' => 'required|in:pending,verified,rejected',
            'transfer_method' => 'required|in:wire,ach,swift,sepa,instant,domestic',
            'transfer_speed' => 'required|in:instant,same_day,next_day,standard',
            'estimated_arrival' => 'nullable|date',
            'actual_arrival' => 'nullable|date',
            'external_reference' => 'nullable|string|max:255',
            'ip_address' => 'nullable|ip',
            'created_at' => 'nullable|date',
        ]);

        try {
            // Prepare transfer data
            $transferData = $request->all();
            $transferData['type'] = 'transfer';

            // Update transfer using service
            $this->transactionService->updateTransaction($transfer, $transferData);

            // Log the transfer update
            activity()
                ->performedOn($transfer)
                ->log("Transfer updated: {$transfer->transaction_id} - {$transfer->transfer_method}");

            return redirect()->route('admin.transfers.show', $transfer)
                ->with('success', 'Transfer updated successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to update transfer: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Remove the specified transfer from storage.
     */
    public function destroy(Transaction $transfer)
    {
        $transferId = $transfer->transaction_id;
        $transfer->delete();

        // Log the transfer deletion
        activity()
            ->log("Transfer deleted: {$transferId}");

        return redirect()->route('admin.transfers.index')
            ->with('success', 'Transfer deleted successfully.');
    }

    /**
     * Approve a transfer.
     */
    public function approve(Transaction $transfer)
    {
        // Ensure this is a transfer transaction
        if ($transfer->type !== 'transfer') {
            abort(404, 'Transfer not found.');
        }

        try {
            $this->transactionService->approveTransaction($transfer);

            activity()
                ->performedOn($transfer)
                ->log("Transfer approved: {$transfer->transaction_id}");

            return redirect()->back()
                ->with('success', 'Transfer approved successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to approve transfer: ' . $e->getMessage());
        }
    }

    /**
     * Reject a transfer.
     */
    public function reject(Transaction $transfer)
    {
        // Ensure this is a transfer transaction
        if ($transfer->type !== 'transfer') {
            abort(404, 'Transfer not found.');
        }

        try {
            $this->transactionService->rejectTransaction($transfer, 'Admin rejection');

            activity()
                ->performedOn($transfer)
                ->log("Transfer rejected: {$transfer->transaction_id}");

            return redirect()->back()
                ->with('success', 'Transfer rejected successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to reject transfer: ' . $e->getMessage());
        }
    }

    /**
     * Reverse a transfer.
     */
    public function reverse(Transaction $transfer)
    {
        // Ensure this is a transfer transaction
        if ($transfer->type !== 'transfer') {
            abort(404, 'Transfer not found.');
        }

        try {
            $this->transactionService->reverseTransaction($transfer, 'Admin reversal');

            activity()
                ->performedOn($transfer)
                ->log("Transfer reversed: {$transfer->transaction_id}");

            return redirect()->back()
                ->with('success', 'Transfer reversed successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to reverse transfer: ' . $e->getMessage());
        }
    }

    /**
     * Mark transfer as arrived.
     */
    public function markArrived(Transaction $transfer)
    {
        // Ensure this is a transfer transaction
        if ($transfer->type !== 'transfer') {
            abort(404, 'Transfer not found.');
        }

        if ($transfer->status !== 'processing') {
            return redirect()->back()
                ->with('error', 'Only processing transfers can be marked as arrived.');
        }

        $transfer->update([
            'status' => 'completed',
            'actual_arrival' => now(),
            'completed_at' => now(),
        ]);

        // Update account balances
        $this->transactionService->updateAccountBalances($transfer);

        activity()
            ->performedOn($transfer)
            ->log("Transfer marked as arrived: {$transfer->transaction_id}");

        return redirect()->back()
            ->with('success', 'Transfer marked as arrived successfully.');
    }

    /**
     * Get transfer statistics.
     */
    public function statistics(Request $request)
    {
        $query = Transaction::where('type', 'transfer');

        // Apply date filters
        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $totalTransfers = $query->count();
        $totalAmount = $query->sum('amount');
        $averageAmount = $totalTransfers > 0 ? $totalAmount / $totalTransfers : 0;

        $statusBreakdown = $query->selectRaw('status, COUNT(*) as count, SUM(amount) as total_amount')
            ->groupBy('status')
            ->get()
            ->keyBy('status');

        $methodBreakdown = $query->selectRaw('transfer_method, COUNT(*) as count, SUM(amount) as total_amount')
            ->groupBy('transfer_method')
            ->get()
            ->keyBy('transfer_method');

        $speedBreakdown = $query->selectRaw('transfer_speed, COUNT(*) as count, SUM(amount) as total_amount')
            ->groupBy('transfer_speed')
            ->get()
            ->keyBy('transfer_speed');

        return response()->json([
            'total_transfers' => $totalTransfers,
            'total_amount' => $totalAmount,
            'average_amount' => $averageAmount,
            'status_breakdown' => $statusBreakdown,
            'method_breakdown' => $methodBreakdown,
            'speed_breakdown' => $speedBreakdown,
        ]);
    }
}
