<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Transaction;
use App\Models\Account;
use App\Models\User;
use App\Services\TransactionService;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Str;

class TransactionController extends Controller
{
    protected $transactionService;

    public function __construct(TransactionService $transactionService)
    {
        $this->transactionService = $transactionService;
    }
    /**
     * Display a listing of transactions.
     */
    public function index(Request $request)
    {
        $query = Transaction::with(['user', 'fromAccount', 'toAccount', 'beneficiary']);

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('transaction_id', 'like', "%{$search}%")
                  ->orWhere('reference_number', 'like', "%{$search}%")
                  ->orWhere('external_reference', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhereHas('user', function ($userQuery) use ($search) {
                      $userQuery->where('name', 'like', "%{$search}%")
                               ->orWhere('email', 'like', "%{$search}%");
                  })
                  ->orWhereHas('fromAccount', function ($accountQuery) use ($search) {
                      $accountQuery->where('account_number', 'like', "%{$search}%");
                  })
                  ->orWhereHas('toAccount', function ($accountQuery) use ($search) {
                      $accountQuery->where('account_number', 'like', "%{$search}%");
                  });
            });
        }

        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('category')) {
            $query->where('category', $request->category);
        }

        if ($request->filled('currency')) {
            $query->where('currency', $request->currency);
        }

        if ($request->filled('user_id')) {
            $query->where('user_id', $request->user_id);
        }

        if ($request->filled('from_account_id')) {
            $query->where('from_account_id', $request->from_account_id);
        }

        if ($request->filled('to_account_id')) {
            $query->where('to_account_id', $request->to_account_id);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        if ($request->filled('amount_min')) {
            $query->where('amount', '>=', $request->amount_min);
        }

        if ($request->filled('amount_max')) {
            $query->where('amount', '<=', $request->amount_max);
        }

        // Apply sorting
        $sortBy = $request->get('sort_by', 'created_at');
        $sortDirection = $request->get('sort_direction', 'desc');
        
        $allowedSortFields = ['created_at', 'transaction_id', 'amount', 'status', 'type', 'processed_at', 'completed_at'];
        if (in_array($sortBy, $allowedSortFields)) {
            $query->orderBy($sortBy, $sortDirection);
        }

        $transactions = $query->paginate(10)->withQueryString();

        // Get filter options
        $users = User::select('id', 'name', 'email')->get();
        $accounts = Account::select('id', 'account_number', 'account_name')->get();
        
        return view('admin.transactions.index', compact('transactions', 'users', 'accounts'));
    }

    /**
     * Show the form for creating a new transaction.
     */
    public function create()
    {
        $users = User::select('id', 'name', 'email')->get();
        $accounts = Account::select('id', 'account_number', 'account_name')->get();
        
        return view('admin.transactions.create', compact('users', 'accounts'));
    }

    /**
     * Store a newly created transaction.
     */
    public function store(Request $request)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
            'from_account_id' => 'nullable|exists:accounts,id',
            'to_account_id' => 'nullable|exists:accounts,id',
            'type' => 'required|in:deposit,withdrawal,transfer,payment,fee,interest,refund,exchange',
            'category' => 'required|string|max:100',
            'subcategory' => 'nullable|string|max:100',
            'amount' => 'required|numeric|min:0.01',
            'currency' => 'required|string|size:3',
            'exchange_rate' => 'nullable|numeric|min:0',
            'converted_amount' => 'nullable|numeric|min:0',
            'fee_amount' => 'nullable|numeric|min:0',
            'tax_amount' => 'nullable|numeric|min:0',
            'net_amount' => 'nullable|numeric|min:0',
            'description' => 'required|string|max:500',
            'notes' => 'nullable|string|max:1000',
            'status' => 'required|in:pending,processing,completed,failed,cancelled,reversed',
            'verification_status' => 'required|in:pending,verified,rejected',
            'external_reference' => 'nullable|string|max:255',
            'ip_address' => 'nullable|ip',
        ]);

        try {
            // Validate transaction data using service
            $validationErrors = $this->transactionService->validateTransactionData($request->all());
            if (!empty($validationErrors)) {
                return redirect()->back()
                    ->withErrors(['validation' => $validationErrors])
                    ->withInput();
            }

            // Create transaction using service
            $transaction = $this->transactionService->createTransaction($request->all());

            // Log the transaction creation
            activity()
                ->performedOn($transaction)
                ->log("Transaction created: {$transaction->transaction_id} - {$transaction->type}");

            return redirect()->route('admin.transactions.show', $transaction)
                ->with('success', 'Transaction created successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to create transaction: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display the specified transaction.
     */
    public function show(Transaction $transaction)
    {
        $transaction->load(['user', 'fromAccount', 'toAccount', 'beneficiary']);
        
        return view('admin.transactions.show', compact('transaction'));
    }

    /**
     * Show the form for editing the specified transaction.
     */
    public function edit(Transaction $transaction)
    {
        $users = User::select('id', 'name', 'email')->get();
        $accounts = Account::select('id', 'account_number', 'account_name')->get();
        
        return view('admin.transactions.edit', compact('transaction', 'users', 'accounts'));
    }

    /**
     * Update the specified transaction.
     */
    public function update(Request $request, Transaction $transaction)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
            'from_account_id' => 'nullable|exists:accounts,id',
            'to_account_id' => 'nullable|exists:accounts,id',
            'type' => 'required|in:deposit,withdrawal,transfer,payment,fee,interest,refund,exchange',
            'category' => 'required|string|max:100',
            'subcategory' => 'nullable|string|max:100',
            'amount' => 'required|numeric|min:0.01',
            'currency' => 'required|string|size:3',
            'exchange_rate' => 'nullable|numeric|min:0',
            'converted_amount' => 'nullable|numeric|min:0',
            'fee_amount' => 'nullable|numeric|min:0',
            'tax_amount' => 'nullable|numeric|min:0',
            'net_amount' => 'nullable|numeric|min:0',
            'description' => 'required|string|max:500',
            'notes' => 'nullable|string|max:1000',
            'status' => 'required|in:pending,processing,completed,failed,cancelled,reversed',
            'verification_status' => 'required|in:pending,verified,rejected',
            'external_reference' => 'nullable|string|max:255',
            'ip_address' => 'nullable|ip',
            'created_at' => 'nullable|date',
        ]);

        $oldStatus = $transaction->status;

        $transaction->update([
            'user_id' => $request->user_id,
            'from_account_id' => $request->from_account_id,
            'to_account_id' => $request->to_account_id,
            'type' => $request->type,
            'category' => $request->category,
            'subcategory' => $request->subcategory,
            'amount' => $request->amount,
            'currency' => $request->currency,
            'exchange_rate' => $request->exchange_rate,
            'converted_amount' => $request->converted_amount,
            'fee_amount' => $request->fee_amount,
            'tax_amount' => $request->tax_amount,
            'net_amount' => $request->net_amount,
            'description' => $request->description,
            'notes' => $request->notes,
            'status' => $request->status,
            'verification_status' => $request->verification_status,
            'external_reference' => $request->external_reference,
            'ip_address' => $request->ip_address,
            'created_at' => $request->created_at ? \Carbon\Carbon::createFromFormat('Y-m-d\TH:i', $request->created_at) : $transaction->created_at,
            'processed_at' => $request->status === 'completed' && $oldStatus !== 'completed' ? now() : $transaction->processed_at,
            'completed_at' => $request->status === 'completed' && $oldStatus !== 'completed' ? now() : $transaction->completed_at,
        ]);

        // Log the transaction update
        activity()
            ->performedOn($transaction)
            ->log("Transaction updated: {$transaction->transaction_id} - Status changed from {$oldStatus} to {$request->status}");

        return redirect()->route('admin.transactions.show', $transaction)
            ->with('success', 'Transaction updated successfully.');
    }

    /**
     * Remove the specified transaction from storage.
     */
    public function destroy(Transaction $transaction)
    {
        $transactionId = $transaction->transaction_id;
        $transaction->delete();

        // Log the transaction deletion
        activity()
            ->log("Transaction deleted: {$transactionId}");

        return redirect()->route('admin.transactions.index')
            ->with('success', 'Transaction deleted successfully.');
    }

    /**
     * Approve a transaction.
     */
    public function approve(Transaction $transaction)
    {
        try {
            $this->transactionService->approveTransaction($transaction);

            activity()
                ->performedOn($transaction)
                ->log("Transaction approved: {$transaction->transaction_id}");

            return redirect()->back()
                ->with('success', 'Transaction approved successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to approve transaction: ' . $e->getMessage());
        }
    }

    /**
     * Reject a transaction.
     */
    public function reject(Transaction $transaction)
    {
        try {
            $this->transactionService->rejectTransaction($transaction, 'Admin rejection');

            activity()
                ->performedOn($transaction)
                ->log("Transaction rejected: {$transaction->transaction_id}");

            return redirect()->back()
                ->with('success', 'Transaction rejected successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to reject transaction: ' . $e->getMessage());
        }
    }

    /**
     * Reverse a transaction.
     */
    public function reverse(Transaction $transaction)
    {
        try {
            $this->transactionService->reverseTransaction($transaction, 'Admin reversal');

            activity()
                ->performedOn($transaction)
                ->log("Transaction reversed: {$transaction->transaction_id}");

            return redirect()->back()
                ->with('success', 'Transaction reversed successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to reverse transaction: ' . $e->getMessage());
        }
    }
}
