<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class PermissionController extends Controller
{
    /**
     * Display a listing of permissions
     */
    public function index(Request $request)
    {
        $query = Permission::withCount('roles', 'users');

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where('name', 'like', "%{$search}%");
        }

        // Filter by category
        if ($request->filled('category')) {
            $query->where('name', 'like', $request->category . '-%');
        }

        // Sort
        $sortBy = $request->get('sort_by', 'name');
        $sortDirection = $request->get('sort_direction', 'asc');
        $query->orderBy($sortBy, $sortDirection);

        $permissions = $query->paginate(20);

        // Get categories for filter
        $categories = Permission::all()->groupBy(function ($permission) {
            return explode('-', $permission->name)[0];
        })->keys()->sort();

        return view('admin.permissions.index', compact('permissions', 'categories'));
    }

    /**
     * Show the form for creating a new permission
     */
    public function create()
    {
        $categories = Permission::all()->groupBy(function ($permission) {
            return explode('-', $permission->name)[0];
        })->keys()->sort();

        return view('admin.permissions.create', compact('categories'));
    }

    /**
     * Store a newly created permission
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255|unique:permissions,name',
            'guard_name' => 'required|string|max:255',
            'description' => 'nullable|string|max:500'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            $permission = Permission::create([
                'name' => $request->name,
                'guard_name' => $request->guard_name ?? 'web'
            ]);

            return redirect()->route('admin.permissions.show', $permission)
                ->with('success', 'Permission created successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to create permission: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display the specified permission
     */
    public function show(Permission $permission)
    {
        $permission->load(['roles', 'users']);
        
        // Get roles with this permission
        $roles = $permission->roles()->paginate(10);

        return view('admin.permissions.show', compact('permission', 'roles'));
    }

    /**
     * Show the form for editing the permission
     */
    public function edit(Permission $permission)
    {
        $categories = Permission::all()->groupBy(function ($permission) {
            return explode('-', $permission->name)[0];
        })->keys()->sort();

        return view('admin.permissions.edit', compact('permission', 'categories'));
    }

    /**
     * Update the specified permission
     */
    public function update(Request $request, Permission $permission)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255|unique:permissions,name,' . $permission->id,
            'guard_name' => 'required|string|max:255',
            'description' => 'nullable|string|max:500'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            $permission->update([
                'name' => $request->name,
                'guard_name' => $request->guard_name ?? 'web'
            ]);

            return redirect()->route('admin.permissions.show', $permission)
                ->with('success', 'Permission updated successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to update permission: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Remove the specified permission
     */
    public function destroy(Permission $permission)
    {
        try {
            // Check if permission is assigned to roles
            if ($permission->roles()->count() > 0) {
                return redirect()->back()
                    ->with('error', 'Cannot delete permission. It is assigned to ' . $permission->roles()->count() . ' role(s).');
            }

            $permission->delete();
            return redirect()->route('admin.permissions.index')
                ->with('success', 'Permission deleted successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to delete permission: ' . $e->getMessage());
        }
    }

    /**
     * Get permission statistics
     */
    public function statistics()
    {
        $stats = [
            'total_permissions' => Permission::count(),
            'total_roles' => Role::count(),
            'permissions_with_roles' => Permission::has('roles')->count(),
            'permissions_without_roles' => Permission::doesntHave('roles')->count(),
            'most_used_permission' => Permission::withCount('roles')->orderBy('roles_count', 'desc')->first(),
            'permissions_by_category' => Permission::all()->groupBy(function ($permission) {
                return explode('-', $permission->name)[0];
            })->map->count(),
            'roles_by_permissions' => Role::withCount('permissions')->orderBy('permissions_count', 'desc')->get()
        ];

        return view('admin.permissions.statistics', compact('stats'));
    }

    /**
     * Bulk assign permissions to role
     */
    public function bulkAssignToRole(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'role_id' => 'required|exists:roles,id',
            'permissions' => 'required|array',
            'permissions.*' => 'exists:permissions,id'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator);
        }

        try {
            $role = Role::findOrFail($request->role_id);
            $permissions = Permission::whereIn('id', $request->permissions)->get();
            $role->givePermissionTo($permissions);

            return redirect()->back()
                ->with('success', 'Permissions assigned to role successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to assign permissions: ' . $e->getMessage());
        }
    }

    /**
     * Bulk remove permissions from role
     */
    public function bulkRemoveFromRole(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'role_id' => 'required|exists:roles,id',
            'permissions' => 'required|array',
            'permissions.*' => 'exists:permissions,id'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator);
        }

        try {
            $role = Role::findOrFail($request->role_id);
            $permissions = Permission::whereIn('id', $request->permissions)->get();
            $role->revokePermissionTo($permissions);

            return redirect()->back()
                ->with('success', 'Permissions removed from role successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to remove permissions: ' . $e->getMessage());
        }
    }
}