<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\KycDocument;
use App\Models\User;
use App\Models\UserProfile;
use App\Services\NotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class KycController extends Controller
{
    protected $notificationService;

    public function __construct(NotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
    }
    /**
     * Display a listing of KYC documents and profiles.
     */
    public function index(Request $request)
    {
        $query = KycDocument::with(['user', 'reviewer'])
            ->withCount(['user']);

        // Search functionality
        if ($request->filled('search')) {
            $search = $request->search;
            $query->whereHas('user', function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }

        // Filter by document type
        if ($request->filled('document_type')) {
            $query->where('document_type', $request->document_type);
        }

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by verification level
        if ($request->filled('verification_level')) {
            $query->where('verification_level', $request->verification_level);
        }

        // Filter by date range
        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        // Sort
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        $documents = $query->paginate(15);

        // Get statistics
        $stats = [
            'total_documents' => KycDocument::count(),
            'pending_documents' => KycDocument::where('status', 'pending')->count(),
            'approved_documents' => KycDocument::where('status', 'approved')->count(),
            'rejected_documents' => KycDocument::where('status', 'rejected')->count(),
            'expired_documents' => KycDocument::where('status', 'expired')->count(),
            'kyc_verified_users' => UserProfile::where('kyc_status', 'verified')->count(),
            'kyc_pending_users' => UserProfile::whereIn('kyc_status', ['pending', 'in_progress'])->count(),
            'aml_flagged_users' => UserProfile::where('aml_status', 'flagged')->count(),
        ];

        return view('admin.kyc.index', compact('documents', 'stats'));
    }

    /**
     * Display the specified KYC document.
     */
    public function show(KycDocument $kyc)
    {
        $kyc->load(['user.profile', 'reviewer']);
        
        // Get user's other KYC documents
        $otherDocuments = KycDocument::where('user_id', $kyc->user_id)
            ->where('id', '!=', $kyc->id)
            ->latest()
            ->get();

        // Get user's profile information
        $userProfile = $kyc->user->profile;

        return view('admin.kyc.show', compact('kyc', 'otherDocuments', 'userProfile'));
    }

    /**
     * Show the form for editing the specified KYC document.
     */
    public function edit(KycDocument $kyc)
    {
        $kyc->load(['user.profile', 'reviewer']);
        
        return view('admin.kyc.edit', compact('kyc'));
    }

    /**
     * Update the specified KYC document.
     */
    public function update(Request $request, KycDocument $kyc)
    {

        $request->validate([
            'status' => 'required|in:pending,approved,rejected,expired',
            'rejection_reason' => 'nullable|string|max:1000',
            'review_notes' => 'nullable|string|max:1000',
            'verification_level' => 'nullable|in:basic,enhanced,full',
            'expires_at' => 'nullable|date|after:today',
        ]);

        try {
            DB::beginTransaction();

            $kyc->update([
                'status' => $request->status,
                'rejection_reason' => $request->rejection_reason,
                'review_notes' => $request->review_notes,
                'verification_level' => $request->verification_level,
                'expires_at' => $request->expires_at,
                'reviewed_by' => Auth::id(),
                'reviewed_at' => now(),
            ]);

            // Update user's KYC status based on document status
            $this->updateUserKycStatus($kyc->user);

            // Log the review
            activity()
                ->performedOn($kyc)
                ->causedBy(Auth::user())
                ->log("KYC document {$kyc->document_type} {$request->status} for user {$kyc->user->name}");

            DB::commit();

            return redirect()->route('admin.kyc.show', $kyc)
                ->with('success', 'KYC document updated successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Failed to update KYC document: ' . $e->getMessage());
        }
    }

    /**
     * Approve the specified KYC document.
     */
    public function approve(Request $request, KycDocument $kyc)
    {

        $request->validate([
            'verification_level' => 'nullable|in:basic,enhanced,full',
            'review_notes' => 'nullable|string|max:1000',
            'expires_at' => 'nullable|date|after:today',
        ]);

        try {
            DB::beginTransaction();

            $kyc->update([
                'status' => 'approved',
                'verification_level' => $request->verification_level ?? 'basic',
                'review_notes' => $request->review_notes,
                'expires_at' => $request->expires_at,
                'reviewed_by' => Auth::id(),
                'reviewed_at' => now(),
            ]);

            // Update user's KYC status
            $this->updateUserKycStatus($kyc->user);

            // Send notification to user
            $this->notificationService->notifyKycApproved($kyc->user);

            // Log the approval
            activity()
                ->performedOn($kyc)
                ->causedBy(Auth::user())
                ->log("KYC document {$kyc->document_type} approved for user {$kyc->user->name}");

            DB::commit();


            return redirect()->route('admin.kyc.show', $kyc)
                ->with('success', 'KYC document approved successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Failed to approve KYC document: ' . $e->getMessage());
        }
    }

    /**
     * Reject the specified KYC document.
     */
    public function reject(Request $request, KycDocument $kyc)
    {

        $request->validate([
            'rejection_reason' => 'required|string|max:1000',
            'review_notes' => 'nullable|string|max:1000',
        ]);

        try {
            DB::beginTransaction();

            $kyc->update([
                'status' => 'rejected',
                'rejection_reason' => $request->rejection_reason,
                'review_notes' => $request->review_notes,
                'reviewed_by' => Auth::id(),
                'reviewed_at' => now(),
            ]);

            // Update user's KYC status
            $this->updateUserKycStatus($kyc->user);

            // Send notification to user
            $this->notificationService->notifyKycRejected($kyc->user, $request->rejection_reason);

            // Log the rejection
            activity()
                ->performedOn($kyc)
                ->causedBy(Auth::user())
                ->log("KYC document {$kyc->document_type} rejected for user {$kyc->user->name}: {$request->rejection_reason}");

            DB::commit();

            return redirect()->route('admin.kyc.show', $kyc)
                ->with('success', 'KYC document rejected successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Failed to reject KYC document: ' . $e->getMessage());
        }
    }

    /**
     * Delete the specified KYC document.
     */
    public function destroy(KycDocument $kyc)
    {
        try {
            // Delete file from storage
            if ($kyc->file_path && Storage::disk('public')->exists($kyc->file_path)) {
                Storage::disk('public')->delete($kyc->file_path);
            }

            // Delete database record
            $kyc->delete();

            // Log the deletion
            activity()
                ->performedOn($kyc)
                ->causedBy(Auth::user())
                ->log("KYC document {$kyc->document_type} deleted for user {$kyc->user->name}");

            return redirect()->route('admin.kyc.index')
                ->with('success', 'KYC document deleted successfully.');

        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to delete KYC document: ' . $e->getMessage());
        }
    }

    /**
     * Download the specified KYC document.
     */
   public function download(KycDocument $kyc)
{
    $filePath = public_path($kyc->file_path);

    if (!$kyc->file_path || !file_exists($filePath)) {
        return redirect()->back()
            ->with('error', 'Document file not found.');
    }

    return response()->download($filePath, $kyc->file_name);
}

/**
 * View the specified KYC document in browser.
 */
public function view(KycDocument $kyc)
{
    $filePath = public_path($kyc->file_path);

    if (!$kyc->file_path || !file_exists($filePath)) {
        return redirect()->back()
            ->with('error', 'Document file not found.');
    }

    $mimeType = $kyc->mime_type ?? mime_content_type($filePath);

    return response()->file($filePath, [
        'Content-Type' => $mimeType,
        'Content-Disposition' => 'inline; filename="' . $kyc->file_name . '"'
    ]);
}


    /**
     * Bulk approve multiple KYC documents.
     */
    public function bulkApprove(Request $request)
    {
        $request->validate([
            'document_ids' => 'required|array',
            'document_ids.*' => 'exists:kyc_documents,id',
            'verification_level' => 'nullable|in:basic,enhanced,full',
        ]);

        try {
            DB::beginTransaction();

            $documents = KycDocument::whereIn('id', $request->document_ids)
                ->where('status', 'pending')
                ->get();

            $approvedCount = 0;
            $notifiedUsers = [];
            foreach ($documents as $document) {
                $document->update([
                    'status' => 'approved',
                    'verification_level' => $request->verification_level ?? 'basic',
                    'reviewed_by' => Auth::id(),
                    'reviewed_at' => now(),
                ]);

                $this->updateUserKycStatus($document->user);
                
                // Send notification to user (only once per user)
                if (!in_array($document->user_id, $notifiedUsers)) {
                    $this->notificationService->notifyKycApproved($document->user);
                    $notifiedUsers[] = $document->user_id;
                }
                
                $approvedCount++;
            }

            // Log the bulk approval
            activity()
                ->causedBy(Auth::user())
                ->log("Bulk approved {$approvedCount} KYC documents");

            DB::commit();

            return redirect()->route('admin.kyc.index')
                ->with('success', "Successfully approved {$approvedCount} KYC documents.");

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Failed to bulk approve documents: ' . $e->getMessage());
        }
    }

    /**
     * Get KYC statistics.
     */
    public function statistics()
    {
        $stats = [
            'total_documents' => KycDocument::count(),
            'pending_documents' => KycDocument::where('status', 'pending')->count(),
            'approved_documents' => KycDocument::where('status', 'approved')->count(),
            'rejected_documents' => KycDocument::where('status', 'rejected')->count(),
            'expired_documents' => KycDocument::where('status', 'expired')->count(),
            
            'kyc_verified_users' => UserProfile::where('kyc_status', 'verified')->count(),
            'kyc_pending_users' => UserProfile::whereIn('kyc_status', ['pending', 'in_progress'])->count(),
            'kyc_rejected_users' => UserProfile::where('kyc_status', 'rejected')->count(),
            
            'aml_clear_users' => UserProfile::where('aml_status', 'clear')->count(),
            'aml_flagged_users' => UserProfile::where('aml_status', 'flagged')->count(),
            'aml_blocked_users' => UserProfile::where('aml_status', 'blocked')->count(),
            
            'compliant_users' => UserProfile::where('compliance_status', 'compliant')->count(),
            'non_compliant_users' => UserProfile::where('compliance_status', 'non_compliant')->count(),
            'pending_review_users' => UserProfile::where('compliance_status', 'pending_review')->count(),
        ];

        // Document type breakdown
        $documentTypeStats = KycDocument::select('document_type', DB::raw('count(*) as count'))
            ->groupBy('document_type')
            ->get()
            ->pluck('count', 'document_type');

        // Status breakdown
        $statusStats = KycDocument::select('status', DB::raw('count(*) as count'))
            ->groupBy('status')
            ->get()
            ->pluck('count', 'status');

        return view('admin.kyc.statistics', compact('stats', 'documentTypeStats', 'statusStats'));
    }

    /**
     * Update user's KYC status based on their documents.
     */
    private function updateUserKycStatus(User $user)
    {
        try {
            $profile = $user->profile;
            if (!$profile) {
                return;
            }

            $approvedDocuments = $user->kycDocuments()->where('status', 'approved')->count();
            $pendingDocuments = $user->kycDocuments()->where('status', 'pending')->count();
            $rejectedDocuments = $user->kycDocuments()->where('status', 'rejected')->count();

            // Determine KYC status based on documents
            if ($rejectedDocuments > 0) {
                $profile->kyc_status = 'rejected';
            } elseif ($approvedDocuments >= 2) { // Require at least 2 approved documents
                $profile->kyc_status = 'verified';
                $profile->kyc_verified_at = now();
            } elseif ($pendingDocuments > 0) {
                $profile->kyc_status = 'under_review';
            } else {
                $profile->kyc_status = 'pending';
            }

            $profile->save();
            
        } catch (\Exception $e) {
            // Silently fail to not break the main flow
        }
    }
}
