<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Spatie\Permission\Models\Role;
use App\Models\Account;
use App\Models\Transaction;
use App\Models\KycDocument;
use App\Models\ComplianceLog;
use App\Models\VirtualCard;
use App\Models\Beneficiary;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DashboardController extends Controller
{
    /**
     * Display the admin dashboard with comprehensive statistics and data.
     */
    public function index()
    {
        // Get current admin user
        $admin = auth()->user();
        
        // Calculate date ranges
        $today = Carbon::today();
        $thisWeek = Carbon::now()->startOfWeek();
        $thisMonth = Carbon::now()->startOfMonth();
        $lastMonth = Carbon::now()->subMonth()->startOfMonth();
        
        // Core Statistics
        $stats = [
            'total_users' => User::whereDoesntHave('roles', function ($q) {
                $q->whereIn('name', ['admin', 'super-admin']);
            })->count(),
            'active_users' => User::where('customer_status', 'active')->whereDoesntHave('roles', function ($q) {
                $q->whereIn('name', ['admin', 'super-admin']);
            })->count(),
            'pending_users' => User::where('customer_status', 'pending_verification')->whereDoesntHave('roles', function ($q) {
                $q->whereIn('name', ['admin', 'super-admin']);
            })->count(),
            'suspended_users' => User::where('customer_status', 'suspended')->whereDoesntHave('roles', function ($q) {
                $q->whereIn('name', ['admin', 'super-admin']);
            })->count(),
            'total_accounts' => Account::count(),
            'active_accounts' => Account::where('status', 'active')->count(),
            'pending_accounts' => Account::where('verification_status', 'pending')->count(),
            'closed_accounts' => Account::where('status', 'closed')->count(),
            
            'total_transactions' => Transaction::count(),
            'pending_transactions' => Transaction::where('status', 'pending')->count(),
            'completed_transactions' => Transaction::where('status', 'completed')->count(),
            'failed_transactions' => Transaction::where('status', 'failed')->count(),
            
            'kyc_pending' => KycDocument::where('status', 'pending')->count(),
            'kyc_approved' => KycDocument::where('status', 'approved')->count(),
            'kyc_rejected' => KycDocument::where('status', 'rejected')->count(),
            
            'compliance_flags' => ComplianceLog::where('status', 'pending')->count(),
            'virtual_cards' => VirtualCard::where('status', 'active')->count(),
            'beneficiaries' => Beneficiary::where('active', true)->count(),
        ];
        
        // Financial Statistics
        $financialStats = [
            'total_balance' => Account::sum('balance'),
            'total_available_balance' => Account::sum('available_balance'),
            'total_hold_balance' => Account::sum('hold_balance'),
            'today_transactions_volume' => Transaction::whereDate('created_at', $today)->sum('amount'),
            'this_week_transactions_volume' => Transaction::where('created_at', '>=', $thisWeek)->sum('amount'),
            'this_month_transactions_volume' => Transaction::where('created_at', '>=', $thisMonth)->sum('amount'),
        ];
        
        // Growth Statistics (Month over Month)
        $growthStats = [
            'user_growth' => $this->calculateGrowth(
                User::where('created_at', '>=', $lastMonth)->where('created_at', '<', $thisMonth)->count(),
                User::where('created_at', '>=', $thisMonth)->count()
            ),
            'account_growth' => $this->calculateGrowth(
                Account::where('created_at', '>=', $lastMonth)->where('created_at', '<', $thisMonth)->count(),
                Account::where('created_at', '>=', $thisMonth)->count()
            ),
            'transaction_growth' => $this->calculateGrowth(
                Transaction::where('created_at', '>=', $lastMonth)->where('created_at', '<', $thisMonth)->sum('amount'),
                Transaction::where('created_at', '>=', $thisMonth)->sum('amount')
            ),
        ];
        
        // Recent Activity
        $recentActivity = $this->getRecentActivity();
        
        // Transaction Trends (Last 7 days)
        $transactionTrends = $this->getTransactionTrends();
        
        // Account Types Distribution
        $accountTypesDistribution = Account::select('account_type', DB::raw('count(*) as count'))
            ->groupBy('account_type')
            ->get()
            ->pluck('count', 'account_type');
        
        // Currency Distribution
        $currencyDistribution = Account::select('currency', DB::raw('count(*) as count'))
            ->groupBy('currency')
            ->get()
            ->pluck('count', 'currency');
        
        // Top Users by Transaction Volume
        $topUsersByVolume = User::withCount(['transactions as total_transactions'])
            ->withSum('transactions as total_volume', 'amount')
            ->orderBy('total_volume', 'desc')
            ->limit(5)
            ->get();
        
        // Pending Actions
        $pendingActions = [
            'kyc_reviews' => KycDocument::where('status', 'pending')->count(),
            'account_approvals' => Account::where('verification_status', 'pending')->count(),
            'transaction_approvals' => Transaction::where('status', 'pending')->count(),
            'compliance_reviews' => ComplianceLog::where('status', 'pending')->count(),
        ];
        
        return view('admin.dashboard', compact(
            'admin',
            'stats',
            'financialStats',
            'growthStats',
            'recentActivity',
            'transactionTrends',
            'accountTypesDistribution',
            'currencyDistribution',
            'topUsersByVolume',
            'pendingActions'
        ));
    }
    
    /**
     * Get recent activity for the dashboard.
     */
    private function getRecentActivity()
    {
        $activities = collect();
        
        // Recent user registrations
        $recentUsers = User::with('profile')
            ->latest()
            ->limit(3)
            ->get()
            ->map(function ($user) {
                return [
                    'type' => 'user_registration',
                    'title' => 'New user registration',
                    'description' => $user->name . ' registered for an account',
                    'time' => $user->created_at,
                    'icon' => 'user-plus',
                    'icon_color' => 'blue',
                ];
            });
        
        // Recent KYC approvals
        $recentKycApprovals = KycDocument::with('user')
            ->where('status', 'approved')
            ->latest('reviewed_at')
            ->limit(3)
            ->get()
            ->map(function ($kyc) {
                return [
                    'type' => 'kyc_approved',
                    'title' => 'KYC approved',
                    'description' => $kyc->user->name . '\'s documents verified',
                    'time' => $kyc->reviewed_at,
                    'icon' => 'check-circle',
                    'icon_color' => 'green',
                ];
            });
        
        // Recent large transactions
        $recentLargeTransactions = Transaction::with(['fromAccount.user', 'toAccount.user'])
            ->where('amount', '>', 10000)
            ->latest()
            ->limit(3)
            ->get()
            ->map(function ($transaction) {
                return [
                    'type' => 'large_transaction',
                    'title' => 'Large transaction',
                    'description' => '$' . number_format($transaction->amount) . ' transfer',
                    'time' => $transaction->created_at,
                    'icon' => 'arrow-right-left',
                    'icon_color' => 'yellow',
                ];
            });
        
        // Recent compliance flags
        $recentComplianceFlags = ComplianceLog::with('user')
            ->where('status', 'pending')
            ->latest()
            ->limit(3)
            ->get()
            ->map(function ($log) {
                return [
                    'type' => 'compliance_flag',
                    'title' => 'Compliance flag',
                    'description' => $log->description,
                    'time' => $log->created_at,
                    'icon' => 'exclamation-triangle',
                    'icon_color' => 'red',
                ];
            });
        
        // Merge and sort all activities
        $activities = $activities
            ->merge($recentUsers)
            ->merge($recentKycApprovals)
            ->merge($recentLargeTransactions)
            ->merge($recentComplianceFlags)
            ->sortByDesc('time')
            ->take(10);
        
        return $activities;
    }
    
    /**
     * Get transaction trends for the last 7 days.
     */
    private function getTransactionTrends()
    {
        $trends = [];
        
        for ($i = 6; $i >= 0; $i--) {
            $date = Carbon::now()->subDays($i);
            $trends[] = [
                'date' => $date->format('M j'),
                'transactions' => Transaction::whereDate('created_at', $date)->count(),
                'volume' => Transaction::whereDate('created_at', $date)->sum('amount'),
            ];
        }
        
        return $trends;
    }
    
    /**
     * Calculate growth percentage between two values.
     */
    private function calculateGrowth($previous, $current)
    {
        if ($previous == 0) {
            return $current > 0 ? 100 : 0;
        }
        
        return round((($current - $previous) / $previous) * 100, 1);
    }
    
    /**
     * Get dashboard analytics data for AJAX requests.
     */
    public function analytics(Request $request)
    {
        $period = $request->get('period', '7d'); // 7d, 30d, 90d, 1y
        
        $startDate = match($period) {
            '7d' => Carbon::now()->subDays(7),
            '30d' => Carbon::now()->subDays(30),
            '90d' => Carbon::now()->subDays(90),
            '1y' => Carbon::now()->subYear(),
            default => Carbon::now()->subDays(7),
        };
        
        $analytics = [
            'user_registrations' => User::where('created_at', '>=', $startDate)
                ->selectRaw('DATE(created_at) as date, COUNT(*) as count')
                ->groupBy('date')
                ->orderBy('date')
                ->get(),
            
            'transaction_volume' => Transaction::where('created_at', '>=', $startDate)
                ->selectRaw('DATE(created_at) as date, SUM(amount) as volume, COUNT(*) as count')
                ->groupBy('date')
                ->orderBy('date')
                ->get(),
            
            'account_creations' => Account::where('created_at', '>=', $startDate)
                ->selectRaw('DATE(created_at) as date, COUNT(*) as count')
                ->groupBy('date')
                ->orderBy('date')
                ->get(),
        ];
        
        return response()->json($analytics);
    }
}
