<?php

namespace App\Helpers;

use App\Models\Setting;
use Illuminate\Support\Facades\Cache;

class SettingsHelper
{
    /**
     * Cache key for settings
     */
    const CACHE_KEY = 'settings_cache';
    const CACHE_TTL = 60; // 1 minute

    /**
     * Get a setting value by key and group
     */
    public static function get(string $key, string $group = null, $default = null)
    {
        $cacheKey = $group ? "{$group}.{$key}" : $key;
        
        return Cache::remember(
            self::CACHE_KEY . '.' . $cacheKey,
            self::CACHE_TTL,
            function () use ($key, $group, $default) {
                $query = Setting::where('key', $key);
                
                if ($group) {
                    $query->where('group', $group);
                }
                
                $setting = $query->first();
                
                if (!$setting) {
                    return $default;
                }
                
                return self::castValue($setting->value, $setting->type);
            }
        );
    }

    /**
     * Get all settings for a specific group
     */
    public static function getGroup(string $group, array $defaults = [])
    {
        return Cache::remember(
            self::CACHE_KEY . '.group.' . $group,
            self::CACHE_TTL,
            function () use ($group, $defaults) {
                $settings = Setting::where('group', $group)->get();
                $result = [];
                
                foreach ($settings as $setting) {
                    $result[$setting->key] = self::castValue($setting->value, $setting->type);
                }
                
                return array_merge($defaults, $result);
            }
        );
    }

    /**
     * Get all public settings
     */
    public static function getPublic()
    {
        return Cache::remember(
            self::CACHE_KEY . '.public',
            self::CACHE_TTL,
            function () {
                return Setting::where('is_public', 1)
                    ->get()
                    ->mapWithKeys(function ($setting) {
                        return [$setting->key => self::castValue($setting->value, $setting->type)];
                    })
                    ->toArray();
            }
        );
    }

    /**
     * Set a setting value
     */
    public static function set(string $key, $value, string $type = 'string', string $group = 'general', string $description = null, bool $isPublic = false)
    {
        $setting = Setting::updateOrCreate(
            ['key' => $key, 'group' => $group],
            [
                'value' => $value,
                'type' => $type,
                'description' => $description,
                'is_public' => $isPublic,
                'updated_at' => now(),
            ]
        );

        // Clear related caches
        self::clearCache($key, $group);
        
        return $setting;
    }

    /**
     * Delete a setting
     */
    public static function delete(string $key, string $group = null)
    {
        $query = Setting::where('key', $key);
        
        if ($group) {
            $query->where('group', $group);
        }
        
        $deleted = $query->delete();
        
        if ($deleted) {
            self::clearCache($key, $group);
        }
        
        return $deleted > 0;
    }

    /**
     * Clear all settings cache
     */
    public static function clearAllCache()
    {
        Cache::forget(self::CACHE_KEY . '.public');
        
        // Clear group caches
        $groups = Setting::distinct()->pluck('group');
        foreach ($groups as $group) {
            Cache::forget(self::CACHE_KEY . '.group.' . $group);
        }
        
        // Clear individual setting caches
        $settings = Setting::all();
        foreach ($settings as $setting) {
            $cacheKey = $setting->group ? "{$setting->group}.{$setting->key}" : $setting->key;
            Cache::forget(self::CACHE_KEY . '.' . $cacheKey);
        }
    }

    /**
     * Clear cache for specific setting
     */
    public static function clearCache(string $key, string $group = null)
    {
        $cacheKey = $group ? "{$group}.{$key}" : $key;
        Cache::forget(self::CACHE_KEY . '.' . $cacheKey);
        
        if ($group) {
            Cache::forget(self::CACHE_KEY . '.group.' . $group);
        }
        
        Cache::forget(self::CACHE_KEY . '.public');
    }

    /**
     * Cast value to appropriate type
     */
    private static function castValue($value, string $type)
    {
        switch ($type) {
            case 'boolean':
                return (bool) $value;
            case 'integer':
                return (int) $value;
            case 'float':
                return (float) $value;
            case 'array':
                return is_string($value) ? json_decode($value, true) : $value;
            case 'json':
                return is_string($value) ? json_decode($value, true) : $value;
            default:
                return $value;
        }
    }

    // Convenience methods for specific groups

    /**
     * Get bank settings
     */
    public static function bank(string $key = null, $default = null)
    {
        if ($key) {
            return self::get($key, 'bank', $default);
        }
        
        return self::getGroup('bank', [
            'bank_name' => 'Cycloverse Bank',
            'bank_code' => 'CYB',
            'swift_code' => 'CYBKENA',
            'bank_address' => 'Nairobi, Kenya',
            'phone_number' => '+254 700 000 000',
            'email' => 'info@cycloversebank.com',
            'website' => 'https://bank.cycloverseworld.info',
            'support_email' => 'support@cycloversebank.com',
            'compliance_email' => 'compliance@cycloversebank.com',
            'logo_url' => '/images/logo.png',
            'favicon_url' => '/images/favicon.ico',
        ]);
    }

    /**
     * Get compliance settings
     */
    public static function compliance(string $key = null, $default = null)
    {
        if ($key) {
            return self::get($key, 'compliance', $default);
        }
        
        return self::getGroup('compliance', [
            'kyc_required' => true,
            'kyc_verification_required' => true,
            'aml_monitoring' => true,
            'transaction_monitoring' => true,
            'suspicious_activity_threshold' => 10000.00,
        ]);
    }

    /**
     * Check if KYC is required
     */
    public static function isKycRequired(): bool
    {
        return (bool) self::compliance('kyc_required', true);
    }

    /**
     * Check if AML monitoring is enabled
     */
    public static function isAmlMonitoringEnabled(): bool
    {
        return (bool) self::compliance('aml_monitoring', true);
    }

    /**
     * Check if transaction monitoring is enabled
     */
    public static function isTransactionMonitoringEnabled(): bool
    {
        return (bool) self::compliance('transaction_monitoring', true);
    }

    /**
     * Get suspicious activity threshold
     */
    public static function getSuspiciousActivityThreshold(): float
    {
        return (float) self::compliance('suspicious_activity_threshold', 10000.00);
    }

    /**
     * Get limits settings
     */
    public static function limits(string $key = null, $default = null)
    {
        if ($key) {
            return self::get($key, 'limits', $default);
        }
        
        return self::getGroup('limits', [
            'daily_transfer_limit' => 20000.00,
            'monthly_transfer_limit' => 100000.00,
            'daily_withdrawal_limit' => 5000.00,
            'monthly_withdrawal_limit' => 50000.00,
        ]);
    }

    /**
     * Get notification settings
     */
    public static function notifications(string $key = null, $default = null)
    {
        if ($key) {
            return self::get($key, 'notifications', $default);
        }
        
        return self::getGroup('notifications', [
            'email_verification_enabled' => true,
            'smtp_driver' => 'smtp',
            'smtp_host' => 'smtp.gmail.com',
            'smtp_port' => 587,
            'smtp_username' => '',
            'smtp_password' => '',
            'smtp_encryption' => 'tls',
            'from_email' => 'noreply@bank.cycloverseworld.info',
            'from_name' => 'Cycloverse Bank',
        ]);
    }

    /**
     * Get system settings
     */
    public static function system(string $key = null, $default = null)
    {
        if ($key) {
            return self::get($key, 'system', $default);
        }
        
        return self::getGroup('system', [
            'maintenance_mode' => false,
            'debug_mode' => false,
        ]);
    }

    /**
     * Check if maintenance mode is enabled
     */
    public static function isMaintenanceMode(): bool
    {
        return self::system('maintenance_mode', false);
    }

    /**
     * Check if debug mode is enabled
     */
    public static function isDebugMode(): bool
    {
        return self::system('debug_mode', false);
    }

    /**
     * Get bank name
     */
    public static function bankName(): string
    {
        return self::bank('bank_name', 'Cycloverse Bank');
    }

    /**
     * Get bank logo URL
     */
    public static function bankLogo(): string
    {
        return self::bank('logo_url', '/images/logo.png');
    }

    /**
     * Get bank favicon URL
     */
    public static function bankFavicon(): string
    {
        return self::bank('favicon_url', '/images/favicon.ico');
    }

    /**
     * Get bank phone number
     */
    public static function bankPhone(): string
    {
        return self::bank('phone_number', '+254 700 000 000');
    }

    /**
     * Get bank email
     */
    public static function bankEmail(): string
    {
        return self::bank('email', 'info@cycloversebank.com');
    }

    /**
     * Get bank address
     */
    public static function bankAddress(): string
    {
        return self::bank('bank_address', 'Nairobi, Kenya');
    }

    /**
     * Get bank website
     */
    public static function bankWebsite(): string
    {
        return self::bank('website', 'https://bank.cycloverseworld.info');
    }

    /**
     * Get bank code
     */
    public static function bankCode(): string
    {
        return self::bank('bank_code', 'CYB');
    }

    /**
     * Get bank SWIFT code
     */
    public static function bankSwiftCode(): string
    {
        return self::bank('swift_code', 'CYBKENA');
    }

    /**
     * Get support email
     */
    public static function supportEmail(): string
    {
        return self::bank('support_email', 'support@cycloversebank.com');
    }

    /**
     * Get compliance email
     */
    public static function complianceEmail(): string
    {
        return self::bank('compliance_email', 'compliance@cycloversebank.com');
    }

    /**
     * Get SMTP configuration for mail
     */
    public static function getSmtpConfig(): array
    {
        $notifications = self::notifications();
        
        return [
            'driver' => $notifications['smtp_driver'] ?? 'smtp',
            'host' => $notifications['smtp_host'] ?? 'smtp.gmail.com',
            'port' => $notifications['smtp_port'] ?? 587,
            'username' => $notifications['smtp_username'] ?? '',
            'password' => $notifications['smtp_password'] ?? '',
            'encryption' => $notifications['smtp_encryption'] ?? 'tls',
            'from' => [
                'address' => $notifications['from_email'] ?? 'noreply@bank.cycloverseworld.info',
                'name' => $notifications['from_name'] ?? 'Cycloverse Bank',
            ],
        ];
    }

    /**
     * Check if email verification is enabled
     */
    public static function isEmailVerificationEnabled(): bool
    {
        return self::notifications('email_verification_enabled', true);
    }

    /**
     * Get transaction limits
     */
    public static function getTransactionLimits(): array
    {
        return self::limits();
    }

    /**
     * Get compliance settings
     */
    public static function getComplianceSettings(): array
    {
        return self::compliance();
    }
}
