<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\ComplianceService;
use App\Helpers\SettingsHelper;

class RunAmlMonitoring extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'aml:monitor {--force : Force monitoring even if disabled}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Run automated AML monitoring on recent transactions and users';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        // Check if AML monitoring is enabled
        if (!SettingsHelper::isAmlMonitoringEnabled() && !$this->option('force')) {
            $this->info('AML monitoring is disabled. Use --force to run anyway.');
            return 0;
        }

        $this->info('Starting AML monitoring...');
        
        $complianceService = app(ComplianceService::class);
        $results = $complianceService->runAutomatedMonitoring();

        $this->info('AML monitoring completed!');
        $this->table(
            ['Metric', 'Count'],
            [
                ['Transactions Monitored', $results['transactions_monitored']],
                ['Transactions Flagged', $results['transactions_flagged']],
                ['Users Monitored', $results['users_monitored']],
                ['Users Flagged', $results['users_flagged']],
            ]
        );

        // Show AML statistics
        $stats = $complianceService->getAmlStats();
        $this->info("\nAML Statistics:");
        $this->table(
            ['Metric', 'Count'],
            [
                ['Total Monitored Transactions (24h)', $stats['total_monitored_transactions']],
                ['Flagged Transactions (24h)', $stats['flagged_transactions']],
                ['High Risk Transactions (24h)', $stats['high_risk_transactions']],
                ['Structuring Detected (24h)', $stats['structuring_detected']],
                ['PEP Transactions (24h)', $stats['pep_transactions']],
                ['Suspicious Activity Threshold', '$' . number_format($stats['suspicious_threshold'])],
            ]
        );

        return 0;
    }
}
