<?php

namespace App\Console\Commands;

use App\Models\User;
use Illuminate\Console\Command;

class CreateUserCodes extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'user:create-codes {--force : Force creation even if codes exist}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Create codes for users who have profile and accounts';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Creating codes for eligible users...');

        if ($this->option('force')) {
            // Create codes for all users with profile and accounts
            $users = User::whereHas('profile')
                ->whereHas('accounts')
                ->get();

            $created = 0;
            foreach ($users as $user) {
                $this->createCodesForUser($user);
                $created++;
            }

            $this->info("Created codes for {$created} users.");
        } else {
            // Only create codes for users who don't have them
            $users = User::whereHas('profile')
                ->whereHas('accounts')
                ->whereDoesntHave('code')
                ->get();

            $created = 0;
            foreach ($users as $user) {
                $this->createCodesForUser($user);
                $created++;
            }

            $this->info("Created codes for {$created} new users.");
        }

        return Command::SUCCESS;
    }

    /**
     * Create codes record for user with profile and account.
     */
    private function createCodesForUser(User $user): void
    {
        // Only create codes if user has profile and at least one account
        if ($user->profile && $user->accounts()->exists()) {
            \App\Models\Code::firstOrCreate(
                ['user_id' => $user->id],
                [
                    'cot_code' => 'COT' . strtoupper(\Illuminate\Support\Str::random(6)),
                    'cot_status' => false,
                    'cot_msg' => 'COT code generated',
                    'tax_code' => 'TAX' . strtoupper(\Illuminate\Support\Str::random(6)),
                    'tax_status' => false,
                    'tax_msg' => 'Tax code generated',
                    'imf_code' => 'IMF' . strtoupper(\Illuminate\Support\Str::random(6)),
                    'imf_status' => false,
                    'imf_msg' => 'IMF code generated',
                    'allow_local' => false,
                    'allow_international' => false,
                ]
            );
        }
    }
}
